% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rrpca.R
\name{rrpca}
\alias{rrpca}
\title{Randomized robust principal component analysis (rrpca).}
\usage{
rrpca(A, k = NULL, lamb = NULL, gamma = 1.25, rho = 1.5, maxiter = 50,
  tol = 0.001, svdalg = "auto", p = 10, q = 1, trace = FALSE, ...)
}
\arguments{
\item{A}{array_like \cr
a numeric input matrix (or data frame), with dimensions \eqn{(m, n)}. \cr
If the data contain \eqn{NA}s na.omit is applied.}

\item{k}{int, optional \cr
determines the number of principle components to compute. It is required that \eqn{k} is smaller or equal to
\eqn{n}, but it is recommended that \eqn{k << min(m,n)}.}

\item{lamb}{real, optional \cr
tuning paramter (default \eqn{lamb=max(m,n)^-0.5}).}

\item{gamma}{real, optional \cr
tuning paramter (default \eqn{gamma=1.25}).}

\item{rho}{real, optional \cr
tuning paramter (default \eqn{rho=1.5}).}

\item{maxiter}{int, optional \cr
determines the maximal numbers of iterations (default \eqn{maxiter=20})..}

\item{tol}{real, optional \cr
tolarance paramter for the desired convergence of the algorithm.}

\item{svdalg}{str c('auto', 'rsvd', 'svd'), optional \cr
Determines which algorithm should be used for computing the singular value decomposition.
By default 'auto' is used, which decides whether to use \code{\link{rsvd}} or \code{\link{svd}},
depending on the number of principle components. If \eqn{k < min(n,m)/1.5} randomized svd is used.}

\item{p}{int, optional \cr
oversampling parameter for \eqn{rsvd}  (default \eqn{p=0}), see \code{\link{rsvd}}.}

\item{q}{int, optional \cr
number of power iterations  for \eqn{rsvd} (default \eqn{q=1}), see \code{\link{rsvd}}.}

\item{trace}{bool, optional \cr
print progress.}

\item{...}{arguments passed to or from other methods, see \code{\link{rsvd}}.}

\item{.................}{.}
}
\value{
\code{rrpca} returns a list with class \eqn{rrpca} containing the following components:
   \item{L}{  array_like \cr
             Low-rank component, array of shape \eqn{(m, n)}.
   }
   \item{S}{  array_like \cr
              Sparse component, array of shape \eqn{(m, n)}.
   }

   \item{k}{  int \cr
              target-rank used for the final iteration.
   }

   \item{err}{  vector \cr
              Frobenious error archieved by each iteration.
   }

   \item{.................}{.}
}
\description{
Robust principal components analysis using randomized singular value decomposition.
}
\details{
Robust principal component analysis (RPCA) is a method for the robust seperation of a
a rectangular \eqn{(m,n)} matrix \eqn{A} into a low-rank component \eqn{L} and a
sparse comonent \eqn{S} as follows: \eqn{A=L+S}.
Here we are using the fast randomized accelerated inexact augmented Lagrange multiplier
method (IALM) for obtaining the robust seperation.
}
\note{
...
}
\examples{
library(rsvd)

# Create toy video
# background frame
xy <- seq(-50, 50, length.out=100)
mgrid <- list( x=outer(xy*0,xy,FUN="+"), y=outer(xy,xy*0,FUN="+") )
bg <- 0.1*exp(sin(-mgrid$x**2-mgrid$y**2))
toyVideo <- matrix(rep(c(bg), 100), 100*100, 100)

# add moving object
for(i in 1:90) {
  mobject <- matrix(0, 100, 100)
  mobject[i:(10+i), 45:55] <- 0.2
  toyVideo[,i] =  toyVideo[,i] + c( mobject )
}

# Foreground/Background separation
out <- rrpca(toyVideo, k=1, p=5, q=1, svdalg='rsvd', trace=TRUE)

# Display results of the seperation for the 10th frame
par(mfrow=c(1,4))
image(matrix(bg, ncol=100, nrow=100)) #true background
image(matrix(toyVideo[,10], ncol=100, nrow=100)) # frame
image(matrix(out$L[,10], ncol=100, nrow=100)) # seperated background
image(matrix(out$S[,10], ncol=100, nrow=100)) #seperated foreground
}
\author{
N. Benjamin Erichson, \email{nbe@st-andrews.ac.uk}
}
\references{
\itemize{
  \item  [1] Lin, Zhouchen, Minming Chen, and Yi Ma.
         "The augmented lagrange multiplier method for exact
         recovery of corrupted low-rank matrices." (2010).
         (available at arXiv \url{http://arxiv.org/abs/1009.5055}).
  \item  [2] Candes, Emmanuel J., et al.
         "Robust principal component analysis?."
         Journal of the ACM (JACM) 58.3 (2011).
}
}

