#' search_tweets
#'
#' @description Returns two data frames (tweets data and users data)
#'   using a provided search query.
#'
#' @param q Query to be searched, used to filter and select tweets
#'   to return from Twitter's REST API. Must be a character string not
#'   to exceed maximum of 500 characters. Spaces behave like boolean
#'   "AND" operator. To search for tweets containing at least one of
#'   multiple possible terms, separate each search term with spaces
#'   and "OR" (in caps). For example, the search
#'   \code{q = "data science"} looks for tweets containing both
#'   "data" and "science" anywhere located anywhere in the tweets and
#'   in any order. When "OR" is entered between search terms,
#'   \code{query = "data OR science"}, Twitter's REST API should return
#'   any tweet that contains either "data" or
#'   "science." It is also possible to search for exact phrases using
#'   double quotes. To do this, either wrap single quotes around a
#'   search query using double quotes, e.g.,
#'   \code{q = '"data science"'} or escape each internal double quote
#'   with a single backslash, e.g., \code{q = "\"data science\""}.
#' @param n Integer, specifying the total number of desired tweets to
#'   return. Defaults to 100. Maximum number of tweets returned from
#'   a single token is 18,000. To return more than 18,000 tweets, users
#'   are encouraged to set \code{retryonratelimit} to TRUE. See details
#'   for more information.
#' @param type Character string specifying which type of search
#'   results to return from Twitter's REST API. The current default is
#'   \code{type = "recent"}, other valid types include
#'   \code{type = "mixed"} and \code{type = "popular"}.
#' @param include_rts Logical, indicating whether to include retweets
#'   in search results. Retweets are classified as any tweet
#'   generated by Twitter's built-in "retweet" (recycle arrows)
#'   function. These are distinct from quotes (retweets with
#'   additional text provided from sender) or manual retweets
#'   (old school method of manually entering "RT" into the text
#'   of one's tweets).
#' @param max_id Character string specifying the [oldest] status
#'   id beyond which search results should resume returning.
#'   Especially useful large data returns that require multiple
#'   iterations interrupted by user time constraints. For searches
#'   exceeding 18,000 tweets, users are encouraged to take advantage
#'   of rtweet's internal automation procedures for waiting on
#'   rate limits by setting \code{retryonratelimit} argument to TRUE.
#'   It some cases, it is possible that due to processing time and
#'   rate limits, retreiving several million tweets can take several
#'   hours or even multiple days. In these cases, it would likely be
#'   useful to leverage \code{retryonratelimit} for sets of tweets
#'   and \code{max_id} to allow results to continue where previous
#'   efforts left off.
#' @param usr Logical indicating whether to return a data frame of
#'   users data. Users data is stored as an attribute. To access this
#'   data, see \code{\link{users_data}}. Useful for marginal returns in
#'   memory demand. However, any gains are likely to be negligible as
#'   Twitter's API invariably returns this data anyway. As such, this
#'   defaults to true, see \code{\link{users_data}}.
#' @param parse Logical, indicating whether to return parsed
#'   data.frame, if true, or nested list (fromJSON), if false. By default,
#'   \code{parse = TRUE} saves users from the wreck of time and frustration
#'   associated with disentangling the nasty nested list returned
#'   from Twitter's API (for proof, check rtweet's Github commit history).
#'   As Twitter's APIs are subject to change, this argument would be
#'   especially useful when changes to Twitter's APIs affect performance of
#'   internal parsers. Setting \code{parse = FALSE} also ensures the
#'   maximum amount of possible information is returned. By default, the
#'   rtweet parse process returns nearly all bits of information returned
#'   from Twitter. However, users may occassionally encounter new or
#'   omitted variables. In these rare cases, the nested list object will
#'   be the only way to access these variables.
#' @param token OAuth token. By default \code{token = NULL} fetches a
#'   non-exhausted token from an environment variable. Find
#'   instructions on how to create tokens and setup an environment
#'   variable in the tokens vignette (in r, send \code{?tokens} to
#'   console).
#' @param retryonratelimit Logical indicating whether to wait and
#'   retry when rate limited. This argument is only relevant if the
#'   desired return (n) exceeds the remaining limit of available
#'   requests (assuming no other searches have been conducted in the
#'   past 15 minutes, this limit is 18,000 tweets). Defaults to false.
#'   Set to TRUE to automate process of conducting big searches
#'   (i.e., n > 18000). For many search queries, esp. specific or
#'   specialized searches, there won't be more than
#'   18,000 tweets to return. But for broad, generic, or popular
#'   topics, the total number of tweets within the REST window of
#'   time (7-10 days) can easily reach the millions.
#' @param verbose Logical, indicating whether or not to include
#'   output processing/retrieval messages. Defaults to TRUE. For
#'   larger searches, messages include rough estimates for time
#'   remaining between searches. It should be noted, however, that
#'   these time estimates only describe the amount of time between
#'   searches and not the total time remaining. For large searches
#'   conducted with \code{retryonratelimit} set to TRUE, the
#'   estimated retreival time can be estimated by dividing the number
#'   of requested tweets by 18,000 and then multiplying the quotient
#'   by 15 (token cooldown time, in minutes).
#' @param \dots Futher arguments passed on to \code{make_url}.
#'   All named arguments that do not match the above arguments
#'   (i.e., count, type, etc.) will be built into the request.
#'   To return only English language tweets, for example, use
#'   \code{lang = "en"}. For more options see Twitter's
#'   API documentation.
#' @seealso \url{https://dev.twitter.com/overview/documentation}
#' @details Twitter API documentation recommends limiting searches to
#'   10 keywords and operators. Complex queries may also produce
#'   API errors preventing recovery of information related to
#'   the query.
#'   It should also be noted Twitter's search API does not consist
#'   of an index of all Tweets. At the time of searching, the
#'   search API index includes between only 6-9 days of Tweets.
#'
#'
#'   Number of tweets returned will often be less than what was
#'   specified by the user. This can happen because (a) the search
#'   query did not return many results (the search pool is already
#'   thinned out from the population of tweets to begin with),
#'   (b) because user hitting rate limit for a given token, or (c)
#'   of recent activity (either more tweets, which affect pagination
#'   in returned results or deletion of tweets). To return more than
#'   18,000 tweets in a single call, users must set
#'   \code{retryonratelimit} argument to true. This method relies on
#'   updating the \code{max_id} parameter and waiting for token rate
#'   limits to refresh between searches. As a result, it is possible
#'   to search for 50,000, 100,000, or even 10,000,000 tweets, but
#'   these searches can take hours or even days. At these durations,
#'   it would not be uncommon for connections to timeout. Users are
#'   instead encouraged to breakup data retrieval into smaller chunks
#'   by leveraging \code{retryonratelimit} and then using the
#'   status_id of the oldest tweet as the \code{max_id} to resume
#'   searching where the previous efforts left off.
#'
#'
#' @examples
#' \dontrun{
#' ## search for 1000 tweets mentioning Hillary Clinton
#' hrc <- search_tweets(q = "hillaryclinton", n = 1000)
#'
#' ## data frame where each observation (row) is a different tweet
#' hrc
#'
#' ## users data also retrieved. can access it via users_data()
#' users_data(hrc)
#'
#' ## search for 1000 tweets in English
#' djt <- search_tweets(q = "realdonaldtrump", n = 1000, lang = "en")
#' djt
#' users_data(djt)
#'
#' ## exclude retweets
#' rt <- search_tweets("rstats", n = 500, include_rts = FALSE)
#'
#' ## perform search for lots of tweets
#' rt <- search_tweets("trump OR president OR potus", n = 100000,
#'                     retryonratelimit = TRUE)
#'
#' ## plot time series of tweets frequency
#' ts_plot(rt, by = "mins", theme = "spacegray",
#'         main = "Tweets about Trump")
#'
#' }
#' @return List object with tweets and users each returned as a
#'   data frame.
#' @family tweets
#' @export
search_tweets <- function(q, n = 100,
                          type = "recent",
                          max_id = NULL,
                          include_rts = TRUE,
                          parse = TRUE,
                          usr = TRUE,
                          token = NULL,
                          retryonratelimit = FALSE,
                          verbose = TRUE,
                          ...) {

    ## check token and get rate limit data
    token <- check_token(token, "search/tweets")
    rtlimit <- rate_limit(token, "search/tweets")
    remaining <- rtlimit[["remaining"]] * 100
    reset <- rtlimit[["reset"]]
    units(reset) <- "mins"

    if (any(n <= remaining, !retryonratelimit)) {
        rt <- .search_tweets(
            q = q, n = n,
            type = type,
            max_id = max_id,
            include_rts = include_rts,
            parse = parse,
            usr = usr,
            token = token,
            verbose = verbose,
            ...)
    } else {
        if (identical(remaining, 0)) {
            ntimes <- ceiling((n - remaining) / 18000)
        } else {
            ntimes <- ceiling((n - remaining) / 18000) + 1
        }
        rt <- vector("list", ntimes)
        maxid <- max_id

        for (i in seq_len(ntimes)) {
            ## if rate limited (exhausted token)
            if (identical(remaining, 0)) {
                message(paste0(
                    "retry on rate limit...\n",
                    "waiting about ",
                    round(reset, 0),
                    " minutes..."))
                Sys.sleep(as.double(reset) * 60 + 1)
                remaining <- 180 * 100
            }
            rt[[i]] <- tryCatch(
                .search_tweets(
                    q = q, n = remaining,
                    check = FALSE,
                    type = type,
                    max_id = maxid,
                    include_rts = include_rts,
                    parse = parse,
                    usr = usr,
                    token = token,
                    verbose = verbose, ...),
                error = function(e) return(NULL))
            ## break if error
            if (is.null(rt[[i]])) break
            ## break if final i
            if (i == ntimes) break
            ## get next maxid
            maxid.new <- rt[[i]][["status_id"]][[NROW(rt[[i]])]]
            ## break if new maxid is null, empty, or unchanged
            if (any(is.null(maxid.new),
                    identical(length(maxid.new), 0L),
                    identical(maxid, maxid.new))) break
            ## update maxid value
            maxid <- maxid.new
            ## refresh rate limit data
            rtlimit <- rate_limit(token, "search/tweets")
            remaining <- rtlimit[["remaining"]] * 100
            reset <- rtlimit[["reset"]]
            units(reset) <- "mins"
        }
        ## get users data if applicable
        if (usr) {
            users <- do.call("rbind", users_data(rt))
            rt <- do.call("rbind", rt)
            attr(rt, "users") <- users
        } else {
            rt <- do.call("rbind", rt)
        }
    }
    rt
}


.search_tweets <- function(q, n = 100,
                          check = FALSE,
                          type = "recent",
                          max_id = NULL,
                          include_rts = TRUE,
                          parse = TRUE,
                          usr = TRUE,
                          token = NULL,
                          verbose = TRUE, ...) {

    query <- "search/tweets"
    stopifnot(is_n(n), is.atomic(q), is.atomic(max_id))
    #token <- check_token(token, query)

    n.times <- ceiling(n / 100)

    if (nchar(q) > 500) {
        stop("q cannot exceed 500 characters.", call. = FALSE)
    }

    if (length(type) > 1) {
        stop("can only select one search type. Try type = 'recent'.",
             call. = FALSE)
    }

    if (!isTRUE(tolower(type) %in% c("mixed", "recent", "popular"))) {
        stop("invalid search type - must be mixed, recent, or popular.",
             call. = FALSE)
    }

    if (!include_rts) q <- paste0(q, " -filter:retweets")

    params <- list(q = q,
                   result_type = type,
                   count = 100,
                   max_id = max_id,
                   ...)

    url <- make_url(
        query = query,
        param = params)

    if (verbose) message("Searching for tweets...")

    tw <- scroller(url, n, n.times, type = "search", token)

    if (parse) {
        tw <- parse.piper(tw, usr = usr)
    }

    if (verbose) {
        message("Finished collecting tweets!")
    }
    tw
}


#' search_users
#'
#' @description Returns data frame of users data using a provided
#'   search query.
#'
#' @param q Query to be searched, used in filtering relevant tweets
#'   to return from Twitter's REST API. Should be a character
#'   string not to exceed 500 characters maximum. Spaces are assumed
#'   to function like boolean "AND" operators. To search for tweets
#'   including one of multiple possible terms, separate search terms
#'   with spaces and the word "OR". For example, the search
#'   \code{query = "data science"} searches for tweets using both
#'   "data" and "science" though the words can appear anywhere and
#'   in any order in the tweet. However, when OR is added between
#'   search terms, \code{query = "data OR science"}, Twitter's REST
#'   API should return any tweet that includes either "data" or
#'   "science" appearing in the tweets. At this time, Twitter's users/search
#'   API does not allow complex searches or queries targetting exact phrases
#'   as is allowed by \code{search_tweets}.
#' @param n Numeric, specifying the total number of desired users to
#'   return. Defaults to 100. Maximum number of users returned from
#'   a single search is 1,000.
#' @param parse Logical, indicating whether to return parsed
#'   (data.frames) or nested list (fromJSON) object. By default,
#'   \code{parse = TRUE} saves users from the time
#'   [and frustrations] associated with disentangling the Twitter
#'   API return objects.
#' @param tw Logical indicating whether to return tweets data frame.
#'   Defaults to true.
#' @param token OAuth token. By default \code{token = NULL} fetches a
#'   non-exhausted token from an environment variable. Find instructions
#'   on how to create tokens and setup an environment variable in the
#'   tokens vignette (in r, send \code{?tokens} to console).
#' @param verbose Logical, indicating whether or not to output
#'   processing/retrieval messages.
#' @seealso \url{https://dev.twitter.com/overview/documentation}
#' @examples
#' \dontrun{
#' # search for 1000 tweets mentioning Hillary Clinton
#' pc <- search_users(q = "political communication", n = 1000)
#'
#' # data frame where each observation (row) is a different user
#' pc
#'
#' # tweets data also retrieved. can access it via tweets_data()
#' users_data(hrc)
#' }
#' @return Data frame of users returned by query.
#' @family users
#' @export
search_users <- function(q, n = 20,
                         parse = TRUE,
                         tw = TRUE,
                         token = NULL,
                         verbose = TRUE) {

    query <- "users/search"
    stopifnot(is_n(n), is.atomic(q))
    token <- check_token(token, query)
    if (n > 1000) {
        warning(
            paste0("search only returns up to 1,000 users per ",
                   "unique search. Setting n to 1000..."))
        n <- 1000
    }
    n.times <- ceiling(n / 20)
    if (n.times > 50) n.times <- 50

    if (nchar(q) > 500) {
        stop("q cannot exceed 500 characters.", call. = FALSE)
    }

    params <- list(q = q,
                   count = 20,
                   page = 1)
    url <- make_url(
        query = query,
        param = params)
    if (verbose) message("Searching for users...")

    usr <- vector("list", n.times)
    k <- 0
    nrows <- NULL

    for (i in seq_len(n.times)) {
        r <- tryCatch(
            TWIT(get = TRUE, url, token),
            error = function(e) return(NULL))

        if (is.null(r)) break

        usr[[i]] <- from_js(r)

        if (identical(length(usr[[i]]), 0)) break
        if (isTRUE(is.numeric(NROW(usr[[i]])))) {
            nrows <- NROW(usr[[i]])
        } else {
            if (identical(nrows, 0)) break
            nrows <- 0
        }
        k <- k + nrows
        if (k >= n) break
        url$query$page <- (i + 1L)
    }
    if (parse) {
        usr <- parse.piper.usr(usr, tw = tw)
    }
    if (verbose) {
        message("Finished collecting users!")
    }
    usr
}

count_users_returned <- function(x) {
    length(unique(unlist(lapply(x, function(x) x[["id_str"]]),
                         use.names = FALSE)))
}


next_id <- function(df) {
    if (!all(c("created_at", "status_id") %in% names(df))) {
        stop("wrong data frame - function requires tweets data")
    }
    df <- df[!is.na(df$status_id), ]
    df <- df[order(df$created_at), ]
    df$status_id[1]
}
