% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/auxiliary.R
\name{periodicTrapRule1D}
\alias{periodicTrapRule1D}
\alias{periodicTrapRule2D}
\alias{periodicTrapRule3D}
\alias{integrateSimp1D}
\alias{integrateSimp2D}
\alias{integrateSimp3D}
\title{Quadrature rules in 1D, 2D and 3D}
\usage{
periodicTrapRule1D(fx, endsMatch = FALSE, na.rm = TRUE,
  lengthInterval = 2 * pi)

periodicTrapRule2D(fxy, endsMatch = FALSE, na.rm = TRUE,
  lengthInterval = rep(2 * pi, 2))

periodicTrapRule3D(fxyz, endsMatch = FALSE, na.rm = TRUE,
  lengthInterval = rep(2 * pi, 3))

integrateSimp1D(fx, lengthInterval = 2 * pi, na.rm = TRUE)

integrateSimp2D(fxy, lengthInterval = rep(2 * pi, 2), na.rm = TRUE)

integrateSimp3D(fxyz, lengthInterval = rep(2 * pi, 3), na.rm = TRUE)
}
\arguments{
\item{fx}{vector containing the evaluation of the function to integrate over a uniform grid in \eqn{[x_1,x_2]}.}

\item{endsMatch}{flag to indicate whether the values of the last entries of \code{fx}, \code{fxy} or \code{fxyz} are the ones in the first entries (elements, rows, columns, slices). See examples for usage.}

\item{na.rm}{logical.  Should missing values (including \code{NaN}) be
    removed?}

\item{lengthInterval}{vector containing the lengths of the intervals of integration.}

\item{fxy}{matrix containing the evaluation of the function to integrate over a uniform grid in \eqn{[x_1,x_2]\times[y_1,y_2]}.}

\item{fxyz}{three dimensional array containing the evaluation of the function to integrate over a uniform grid in \eqn{[x_1,x_2]\times[y_1,y_2]\times[z_1,z_2]}.}
}
\value{
The value of the integral.
}
\description{
Quadrature rules for definite integrals over intervals in 1D, \eqn{\int_{x_1}^{x_2} f(x)dx}, rectangles in 2D,\cr \eqn{\int_{x_1}^{x_2}\int_{y_1}^{y_2} f(x,y)dydx} and cubes in 3D, \eqn{\int_{x_1}^{x_2}\int_{y_1}^{y_2}\int_{z_1}^{z_2} f(x,y,z)dzdydx}. The trapezoidal rules assume that the function is periodic, whereas the Simpson rules work for arbitrary functions.
}
\details{
The simple trapezoidal rule has a very good performance for periodic functions in 1D and 2D(order of error ). The higher dimensional extensions are obtained by iterative usage of the 1D rules.
}
\examples{
# In 1D. True value: 3.55099937
N <- 21
grid <- seq(-pi, pi, l = N)
fx <- sin(grid)^2 * exp(cos(grid))
periodicTrapRule1D(fx = fx, endsMatch = TRUE)
periodicTrapRule1D(fx = fx[-N], endsMatch = FALSE)
integrateSimp1D(fx = fx, lengthInterval = 2 * pi)
integrateSimp1D(fx = fx[-N]) # Worse, of course

# In 2D. True value: 22.31159
fxy <- outer(grid, grid, function(x, y) (sin(x)^2 * exp(cos(x)) +
                                         sin(y)^2 * exp(cos(y))) / 2)
periodicTrapRule2D(fxy = fxy, endsMatch = TRUE)
periodicTrapRule2D(fxy = fxy[-N, -N], endsMatch = FALSE)
periodicTrapRule1D(apply(fxy[-N, -N], 1, periodicTrapRule1D))
integrateSimp2D(fxy = fxy)
integrateSimp1D(apply(fxy, 1, integrateSimp1D))

# In 3D. True value: 140.1878
fxyz <- array(fxy, dim = c(N, N, N))
for (i in 1:N) fxyz[i, , ] <- fxy
periodicTrapRule3D(fxyz = fxyz, endsMatch = TRUE)
integrateSimp3D(fxyz = fxyz)
}
\references{
Press, W. H., Teukolsky, S. A., Vetterling, W. T., Flannery, B. P. (1996). \emph{Numerical Recipes in Fortran 77: The Art of Scientific Computing (Vol. 1 of Fortran Numerical Recipes)}. Cambridge University Press, Cambridge.
}
