#' Measurement functions
#'
#' \code{constructs} creates the factors from measurement items by assigning the
#' relevant items to each factor and specifying reflective or formative (composite/causal) measurement models
#'
#' This function conveniently maps measurement items to factors using
#' root name, numbers, and affixes with explicit definition of formative
#' or reflective relationships
#'
#' @param ... Comma separated list of the latent variable measurement specifications, as generated by the
#'    \code{reflective()}, or \code{composite()} methods.
#'
#' @usage
#'   constructs(...)
#'
#' @seealso See \code{\link{composite}}, \code{\link{reflective}}
#'
#' @examples
#'   mobi_mm <- constructs(
#'     reflective("Image",        multi_items("IMAG", 1:5)),
#'     reflective("Expectation",  multi_items("CUEX", 1:3)),
#'     reflective("Quality",      multi_items("PERQ", 1:7)),
#'     reflective("Value",        multi_items("PERV", 1:2)),
#'     reflective("Satisfaction", multi_items("CUSA", 1:3)),
#'     reflective("Complaints",   single_item("CUSCO")),
#'     reflective("Loyalty",      multi_items("CUSL", 1:3))
#'   )
#' @export
constructs <- function(...) {
  return(matrix(c(...), ncol = 3, byrow = TRUE,
                dimnames = list(NULL, c("latent", "measurement", "type"))))
}

#' Reflective construct measurement model specification
#'
#' \code{reflective} creates the reflective measurement model matrix for a specific common-factor,
#' specifying the relevant items of the factor and assigning the relationship of reflective.
#' By definition this construct will be estimated by PLS consistent.
#'
#' This function conveniently maps reflectively defined measurement items to a factor and is estimated
#' using PLS consistent.
#'
#' @param construct_name of construct
#' @param item_names returned by the \code{multi_items} or \code{single_item} functions
#'
#' @usage
#'  reflective(construct_name, item_names)
#'
#' @seealso See \code{\link{composite}}, \code{\link{constructs}}
#'
#' @examples
#'   mobi_mm <- constructs(
#'     reflective("Image",        multi_items("IMAG", 1:5)),
#'     reflective("Expectation",  multi_items("CUEX", 1:3)),
#'     reflective("Quality",      multi_items("PERQ", 1:7)),
#'     reflective("Value",        multi_items("PERV", 1:2)),
#'     reflective("Satisfaction", multi_items("CUSA", 1:3)),
#'     reflective("Complaints",   single_item("CUSCO")),
#'     reflective("Loyalty",      multi_items("CUSL", 1:3))
#'   )
#' @export
reflective <- function(construct_name, item_names) {
  construct_names <- rep(construct_name, length(item_names))
  return(c(rbind(construct_names, item_names, "C")))
}

#' Composite construct measurement model specification
#'
#' \code{composite} creates the composite measurement model matrix for a specific construct,
#' specifying the relevant items of the construct and assigning the relationship of either
#' correlation weights (Mode A) or regression weights (Mode B).
#'
#' This function conveniently maps composite defined measurement items to a construct and is
#' estimated using PLS.
#'
#' @param construct_name of construct
#' @param item_names returned by the \code{multi_items} or \code{single_item} functions
#' @param weights is the relationship between the construct and its items. This can be
#' specified as \code{correlation_weights} or \code{mode_A} for correlation weights (Mode A) or as
#' \code{regression_weights} or \code{mode_B} for regression weights (Mode B). Default is correlation weights.
#'
#' @usage
#'  composite(construct_name, item_names,weights = correlation_weights)
#'
#' @seealso See \code{\link{constructs}}, \code{\link{reflective}}
#'
#' @examples
#'   mobi_mm <- constructs(
#'     composite("Image",        multi_items("IMAG", 1:5), weights = correlation_weights),
#'     composite("Expectation",  multi_items("CUEX", 1:3), weights = mode_A),
#'     composite("Quality",      multi_items("PERQ", 1:7), weights = regression_weights),
#'     composite("Value",        multi_items("PERV", 1:2), weights = mode_B)
#'   )
#' @export
composite <- function(construct_name, item_names, weights = correlation_weights) {
  construct_names <- rep(construct_name, length(item_names))
  # TODO remove the duplicated conditional
  # TODO possibly remove the construct_names object as the construct name should be coerced to fitr the matrix
  if(identical(weights,correlation_weights) | identical(weights,mode_A)) {
    return(c(rbind(construct_names,item_names,"A")))
  }
  if(identical(weights, regression_weights) | identical(weights, mode_B)) {
    return(c(rbind(construct_names,item_names,"B")))
  }
}

# arguments:
#   item_name: root name of all items
#   item_numbers: vector of item numbers
#   ...: optional affix arguments
#     prefix: prefix before each item name
#     mid: insert between item name and numbers
#     suffix: suffix after each ite name
#
# e.g.> multi_items("item", 0:3, prefix="X_", mid=".", suffix="_")


#' Multi-items measurement model specification
#'
#' \code{multi_items} creates a vector of measurment names given the item prefix and number range.
#'
#' @param item_name Prefix name of items
#' @param item_numbers The range of number suffixews for the items
#' @param ... Additional Item names and nubers
#'
#' @usage
#'   multi_items(item_name, item_numbers, ...)
#'
#' @seealso See \code{\link{single_item}}
#'
#' @examples
#'   mobi_mm <- constructs(
#'     composite("Image",        multi_items("IMAG", 1:5), weights = correlation_weights),
#'     composite("Expectation",  multi_items("CUEX", 1:3), weights = mode_A),
#'     composite("Quality",      multi_items("PERQ", 1:7), weights = regression_weights),
#'     composite("Value",        multi_items("PERV", 1:2), weights = mode_B)
#'   )
#' @export
multi_items <- function(item_name, item_numbers, ...) {
  affix <- as.data.frame(list(...))
  paste(affix$prefix, item_name, affix$mid, item_numbers, affix$suffix, sep = "")
}

#' Single-item measurement model specification
#'
#' \code{single_item} specifies a single item name to be assigned to a construct.
#'
#' @param item Name of item
#'
#' @usage
#'   single_item(item)
#'
#' @seealso See \code{\link{multi_items}}
#'
#' @examples
#'   mobi_mm <- constructs(
#'     composite("Image",        multi_items("IMAG", 1:5), weights = correlation_weights),
#'     composite("Expectation",  multi_items("CUEX", 1:3), weights = mode_A),
#'     composite("Quality",      multi_items("PERQ", 1:7), weights = regression_weights),
#'     composite("Value",        single_item("PERV1"), weights = mode_A)
#'   )
#' @export
single_item <- function(item) {
  return(item)
}
