% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_intra_phyglm.R
\name{tree_intra_phyglm}
\alias{tree_intra_phyglm}
\title{Interaction between phylogenetic uncertainty and intraspecific variability - Phylogenetic logistic Regression}
\usage{
tree_intra_phyglm(
  formula,
  data,
  phy,
  Vx = NULL,
  x.transf = NULL,
  n.intra = 10,
  n.tree = 2,
  distrib = "normal",
  track = TRUE,
  btol = 50,
  ...
)
}
\arguments{
\item{formula}{The model formula: \code{response~predictor}.}

\item{data}{Data frame containing species traits and species names as row names.}

\item{phy}{A phylogeny (class 'phylo', see ?\code{ape}).}

\item{Vx}{Name of the column containing the standard deviation or the standard error of the predictor 
variable. When information is not available for one taxon, the value can be 0 or \code{NA}}

\item{x.transf}{Transformation for the predictor variable (e.g. \code{log} or \code{sqrt}). Please use this 
argument instead of transforming data in the formula directly (see also details below).}

\item{n.intra}{Number of times to repeat the analysis generating a random value for response and/or predictor variables.
If NULL, \code{n.intra} = 30}

\item{n.tree}{Number of times to repeat the analysis with n different trees picked 
randomly in the multiPhylo file.
If NULL, \code{n.tree} = 2}

\item{distrib}{A character string indicating which distribution to use to generate a random value for the response 
and/or predictor variables. Default is normal distribution: "normal" (function \code{\link{rnorm}}).
Uniform distribution: "uniform" (\code{\link{runif}})
Warning: we recommend to use normal distribution with Vx or Vy = standard deviation of the mean.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{btol}{Bound on searching space. For details see \code{phyloglm}}

\item{...}{Further arguments to be passed to \code{phyloglm}}
}
\value{
The function \code{tree_intra_phylm} returns a list with the following
components:

\code{formula}: The formula

\code{data}: Original full dataset

\code{sensi.estimates}: Coefficients, aic and the optimised value of the phylogenetic 
parameter (e.g. \code{lambda}) for each regression using a value in the interval of variation and 
a different phylogenetic tree.

\code{N.obs}: Size of the dataset after matching it with tree tips and removing NA's.

\code{stats}: Main statistics for model parameters.\code{CI_low} and \code{CI_high} are the lower 
and upper limits of the 95% confidence interval.

\code{all.stats}: Complete statistics for model parameters.
Fields coded using \code{all} describe statistics due to both intraspecific variation and phylogenetic uncertainty.
Fields coded using \code{intra} describe statistics due to intraspecific variation only.
Fields coded using \code{tree} describe statistics due to phylogenetic uncertainty only.
\code{sd} is the standard deviation. \code{CI_low} and \code{CI_high} are the lower and upper limits 
of the 95% confidence interval.

\code{sp.pb}: Species that caused problems with data transformation (see details above).
}
\description{
Performs Phylogenetic logistic regression evaluating
intraspecific variability in response and/or predictor variables
and uncertainty in trees topology.
}
\details{
This function fits a phylogenetic logistic regression model using \code{\link[phylolm]{phyloglm}} to n trees (\code{n.tree}), 
randomly picked in a multiPhylo file. The regression is also repeated \code{n.intra} times.
At each iteration the function generates a random value for each row in the dataset using the standard deviation 
or errors supplied and assuming a normal or uniform distribution. To calculate means and se for your raw data, 
you can use the \code{summarySE} function from the package \code{Rmisc}.

#' All phylogenetic models from \code{phyloglm} can be used, i.e. \code{BM},
\code{OUfixedRoot}, \code{OUrandomRoot}, \code{lambda}, \code{kappa},
\code{delta}, \code{EB} and \code{trend}. See ?\code{phyloglm} for details.

Currently, this function can only implement simple logistic models (i.e. \eqn{trait~
predictor}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\section{Warning}{
  
When Vy or Vx exceed Y or X, respectively, negative (or null) values can be generated, this might cause problems
for data transformation (e.g. log-transformation). In these cases, the function will skip the simulation. This problem can
be solved by increasing \code{times}, changing the transformation type and/or checking the target species in output$sp.pb.
}

\examples{
# Simulate data
set.seed(6987)
mphy = ape::rmtree(150, N = 30)
x = phylolm::rTrait(n=1,phy=mphy[[1]])
x_sd = rnorm(150,mean = 0.8,sd=0.2)
X = cbind(rep(1,150),x)
y = rbinTrait(n=1,phy=mphy[[1]], beta=c(-1,0.5), alpha=.7 ,X=X)
dat = data.frame(y, x, x_sd)
intra.tree <- tree_intra_phyglm(y ~ x, data = dat, phy = mphy, n.intra = 3, 
                                           n.tree = 3, Vx = "x_sd")
# summary results:
summary(intra.tree)
# Visual diagnostics for phylogenetic uncertainty:
sensi_plot(intra.tree, uncer.type = "all") #or uncer.type = "tree", uncer.type = "intra"
}
\references{
Paterno, G. B., Penone, C. Werner, G. D. A. 
\href{http://doi.wiley.com/10.1111/2041-210X.12990}{sensiPhy: 
An r-package for sensitivity analysis in phylogenetic 
comparative methods.} Methods in Ecology and Evolution 
2018, 9(6):1461-1467

Martinez, P. a., Zurano, J.P., Amado, T.F., Penone, C., Betancur-R, R., 
Bidau, C.J. & Jacobina, U.P. (2015). Chromosomal diversity in tropical reef 
fishes is related to body size and depth range. Molecular Phylogenetics and 
Evolution, 93, 1-4

Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phyloglm}}, \code{\link{tree_phyglm}}, \code{\link{intra_phyglm}},
\code{\link{tree_intra_phylm}}, \code{\link{sensi_plot}}
}
\author{
Caterina Penone & Pablo Ariel Martinez
}
