#' Wrapper for updating plotly with large-sized data
#'
#' @description
#' using this function, figure updater is registered in shiny app.
#' The figure updater receives events in plotly and update the figure.
#' @param session The \code{session} object passed to function given to
#' \code{shinyServer}.
#' @param outputId Character. The outputId of the plotly that will be
#' down-sampled
#' @param relayout_order Named list.
#' The list generated by \code{plotlyjs_relayout},
#' which is obtained using \code{plotly::event_data}.
#' @param sd_obj The \code{shiny_downsampler} instance
#' that contains the original data and used for generating the figure.
#' @param enforce Boolean.
#' It it is \code{TRUE}, the figure will be updated even if
#' \code{relayout_order} is \code{NULL}.
#' @export
#' @examples
#'\donttest{
#' data(noise_fluct)
#' fig <- plot_ly(x = d$x, y = d$y, type = "scatter", mode = "lines")
#'
#' shd <- shiny_downsampler$new(figure = fig)
#'
#' ui <- fluidPage(
#'   plotlyOutput(outputId = "hp", width = "800px", height = "600px")
#' )
#'
#' server <- function(input, output, session) {
#'
#'   output$hp <- renderPlotly(shd$figure)
#'
#'   observeEvent(plotly::event_data("plotly_relayout"),{
#'     updatePlotlyH(session, "hp", plotly::event_data("plotly_relayout"), shd)
#'   })
#'
#' }
#'
#' shinyApp(ui = ui, server = server)
#'}
updatePlotlyH <- function(session, outputId, relayout_order, sd_obj, enforce = FALSE) {

  trace_update_order <- sd_obj$cmpt_new_trace(relayout_order, enforce)

  if (!is.null(trace_update_order) && length(trace_update_order$new_data) > 0) {
    plotlyProxy(outputId, session) %>%
      plotlyProxyInvoke(
        "deleteTraces", unique(trace_update_order$trace_idx_delete) - 1
      ) %>%
      plotlyProxyInvoke(
        "addTraces",
        trace_update_order$new_data
      )
  }
}
