#' Enable/disable an input element
#'
#' Enable or disable an input element. A disabled element is not usable and
#' not clickable, while an enabled element (default) can receive user input.
#' Any shiny input tag can be used with these functions.\cr\cr
#' \strong{\code{enable}} enables an input, \strong{\code{disable}} disabled
#' an input,\strong{\code{toggleState}} enables an input if it is disabled
#' and disables an input if it is already enabled.\cr\cr
#' If \code{condition} is given to \code{toggleState}, that condition will be used
#' to determine if to enable or disable the input. The element will be enabled if
#' the condition evaluates to \code{TRUE} and disabled otherwise. If you find
#' yourself writing code such as \code{if (test()) enable(id) else disable(id)}
#' then you can use \code{toggleState} instead: \code{toggleState(id, test())}.
#'
#' @param id The id of the input element/Shiny tag
#' @param condition An optional argument to \code{toggleState}, see 'Details' below.
#' @param selector Query selector of the elements to target.  Ignored if the \code{id}
#' argument is given. For example, to disable all text inputs, use
#' \code{selector = "input[type='text']"}
#' @seealso \code{\link[shinyjs]{useShinyjs}},
#' \code{\link[shinyjs]{runExample}}
#' \code{\link[shinyjs]{disabled}}
#' @note \code{shinyjs} must be initialized with a call to \code{useShinyjs()}
#' in the app's ui.
#' @examples
#' if (interactive()) {
#'   library(shiny)
#'
#'   shinyApp(
#'     ui = fluidPage(
#'       useShinyjs(),  # Set up shinyjs
#'       actionButton("btn", "Click me"),
#'       textInput("element", "Watch what happens to me")
#'     ),
#'     server = function(input, output) {
#'       observeEvent(input$btn, {
#'         # Change the following line for more examples
#'         toggleState("element")
#'       })
#'     }
#'   )
#' }
#' \dontrun{
#'   # The shinyjs function call in the above app can be replaced by
#'   # any of the following examples to produce similar Shiny apps
#'   toggleState(id = "element")
#'   enable("element")
#'   disable("element")
#'
#'   # Similarly, the "element" text input can be changed to many other
#'   # input tags, such as the following examples
#'   actionButton("element", "I'm a button")
#'   fileInput("element", "Choose a file")
#'   selectInput("element", "I'm a select box", 1:10)
#' }
#'
#' ## toggleState can be given an optional `condition` argument, which
#' ## determines if to enable or disable the input
#' if (interactive()) {
#'   shinyApp(
#'     ui = fluidPage(
#'       useShinyjs(),
#'       textInput("text", "Please type at least 3 characters"),
#'       actionButton("element", "Submit")
#'     ),
#'     server = function(input, output) {
#'       observe({
#'         toggleState(id = "element", condition = nchar(input$text) >= 3)
#'       })
#'     }
#'   )
#' }
#' @name stateFuncs
NULL

#' @export
#' @rdname stateFuncs
enable <- function(id, selector) {
  fxn <- "enable"
  params <- as.list(match.call())[-1]
  jsFuncHelper(fxn, params)
}

#' @export
#' @rdname stateFuncs
disable <- function(id, selector) {
  fxn <- "disable"
  params <- as.list(match.call())[-1]
  jsFuncHelper(fxn, params)
}

#' @export
#' @rdname stateFuncs
toggleState <- function(id, condition, selector) {
  fxn <- "toggleState"
  params <- as.list(match.call())[-1]
  jsFuncHelper(fxn, params)
}
