% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_accuracy.R
\name{sits_accuracy}
\alias{sits_accuracy}
\alias{sits_accuracy.sits}
\alias{sits_accuracy.class_vector_cube}
\alias{sits_accuracy.class_cube}
\alias{sits_accuracy.raster_cube}
\alias{sits_accuracy.derived_cube}
\alias{sits_accuracy.tbl_df}
\alias{sits_accuracy.default}
\title{Assess classification accuracy}
\usage{
sits_accuracy(data, ...)

\method{sits_accuracy}{sits}(data, ...)

\method{sits_accuracy}{class_vector_cube}(data, ..., prediction_attr, reference_attr)

\method{sits_accuracy}{class_cube}(data, ..., validation, method = "olofsson")

\method{sits_accuracy}{raster_cube}(data, ...)

\method{sits_accuracy}{derived_cube}(data, ...)

\method{sits_accuracy}{tbl_df}(data, ...)

\method{sits_accuracy}{default}(data, ...)
}
\arguments{
\item{data}{Either a data cube with classified images or
a set of time series}

\item{\dots}{Specific parameters}

\item{prediction_attr}{Name of the column of the segments object
that contains the predicted values
(only for vector class cubes)}

\item{reference_attr}{Name of the column of the segments object
that contains the reference values
(only for vector class cubes)}

\item{validation}{Samples for validation (see below)
Only required when data is a raster class cube.}

\item{method}{A character with 'olofsson' or 'pixel' to compute
accuracy (only for raster class cubes)}
}
\value{
A list of lists: The error_matrix, the class_areas, the unbiased
estimated areas, the standard error areas, confidence interval 95% areas,
and the accuracy (user, producer, and overall), or NULL if the data is empty.
The result is assigned to class "sits_accuracy" and can be visualized
directly on the screen.
}
\description{
This function calculates the accuracy of the classification
result. The input is either a set of classified time series or a classified
data cube. Classified time series are produced
by \code{\link[sits]{sits_classify}}.
Classified images are generated using \code{\link[sits]{sits_classify}}
followed by \code{\link[sits]{sits_label_classification}}.

For a set of time series, \code{sits_accuracy} creates a confusion matrix and
calculates the resulting statistics using package \code{caret}. For a
classified image, the function uses an area-weighted technique
proposed by Olofsson et al. according to references [1-3] to produce reliable
accuracy estimates at 95\% confidence level. In both cases, it provides
an accuracy assessment of the classified,
including Overall Accuracy, Kappa, User's Accuracy, Producer's Accuracy
and error matrix (confusion matrix).
}
\note{
The `validation` data needs to contain the following columns: "latitude",
 "longitude", "start_date", "end_date", and "label". It can be either a
 path to a CSV file, a sits tibble, a data frame, or an sf object.

When `validation` is an sf object, the columns "latitude" and "longitude" are
not required as the locations are extracted from the geometry column. The
`centroid` is calculated before extracting the location values for any
geometry type.
}
\examples{
if (sits_run_examples()) {
    # show accuracy for a set of samples
    train_data <- sits_sample(samples_modis_ndvi, frac = 0.5)
    test_data <- sits_sample(samples_modis_ndvi, frac = 0.5)
    rfor_model <- sits_train(train_data, sits_rfor())
    points_class <- sits_classify(
        data = test_data, ml_model = rfor_model
    )
    acc <- sits_accuracy(points_class)

    # show accuracy for a data cube classification
    # create a random forest model
    rfor_model <- sits_train(samples_modis_ndvi, sits_rfor())
    # create a data cube from local files
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6.1",
        data_dir = data_dir
    )
    # classify a data cube
    probs_cube <- sits_classify(
        data = cube, ml_model = rfor_model, output_dir = tempdir()
    )
    # label the probability cube
    label_cube <- sits_label_classification(
        probs_cube,
        output_dir = tempdir()
    )
    # obtain the ground truth for accuracy assessment
    ground_truth <- system.file("extdata/samples/samples_sinop_crop.csv",
        package = "sits"
    )
    # make accuracy assessment
    as <- sits_accuracy(label_cube, validation = ground_truth)
}
}
\references{
[1] Olofsson, P., Foody, G.M., Stehman, S.V., Woodcock, C.E. (2013).
Making better use of accuracy data in land change studies: Estimating
accuracy and area and quantifying uncertainty using stratified estimation.
Remote Sensing of Environment, 129, pp.122-131.

[2] Olofsson, P., Foody G.M., Herold M., Stehman, S.V.,
Woodcock, C.E., Wulder, M.A. (2014)
Good practices for estimating area and assessing accuracy of land change.
Remote Sensing of Environment, 148, pp. 42-57.

[3] FAO, Map Accuracy Assessment and Area Estimation: A Practical Guide.
National forest monitoring assessment working paper No.46/E, 2016.
}
\author{
Rolf Simoes, \email{rolfsimoes@gmail.com}

Alber Sanchez, \email{alber.ipia@inpe.br}
}
