% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_story_clusters.R
\name{get_story_clusters}
\alias{get_story_clusters}
\title{Find clusters of similar stories}
\usage{
get_story_clusters(
  collection = NULL,
  weights = list(choice = 3, major = 2, minor = 1),
  explicit = TRUE,
  min_freq = 1,
  min_size = 3,
  blacklist = NULL
)
}
\arguments{
\item{collection}{A \code{\link[=Collection]{Collection()}} class object.

If \code{NULL}, the collection of all stories in the actively loaded
\acronym{LTO} version is used.}

\item{weights}{A list assigning nonnegative weights to choice, major, and
minor theme levels. The default weighting
\code{list(choice = 3, major = 2, minor = 1)} counts each \emph{choice} usage
three times, each \emph{major} theme usage twice, and each \emph{minor}
theme usage once. Use the uniform weighting
\code{list(choice = 1, major = 1, minor = 1)} weights theme usages equally
regardless of level. At least one weight must be positive.}

\item{explicit}{Set to \code{FALSE} to include ancestor themes of the explicit
thematic annotations.}

\item{min_freq}{Drop themes occurring less than this number of times from
the analysis. The default \code{min_freq=1} results in no themes are discarded.}

\item{min_size}{Minimum cluster size. The default is \code{min_size=3}.}

\item{blacklist}{A \code{\link[=Themeset]{Themeset()}} class object. A themeset containing themes
to be dropped from the analysis.

If \code{NULL}, no themes are filtered.}
}
\value{
Returns a \code{\link[tibble]{tibble}} with \code{r} rows (story
clusters) and 4 columns:
\tabular{ll}{
\code{cluster_id}: \tab Story cluster integer ID\cr
\code{stories}: \tab A tibble of stories comprising the cluster\cr
\code{themes}: \tab A tibble of themes common to the clustered stories\cr
\code{size}: \tab Number of stories in the cluster\cr
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{get_story_clusters} classifies the stories in a collection according to
thematic similarity.
}
\details{
The input collection of \code{n} stories, \eqn{S[1], \ldots, S[n]}, is
represented as a weighted bag-of-words, where each \emph{choice} theme in
story \eqn{S[j] (j=1, \ldots, n)} is counted \code{weights$choice} times,
each \emph{major} theme \code{weights$major} times, and each \emph{minor}
theme \code{weights$choice} times.

The function classifies the stories according to thematic similarity
using the Iterative Signature Algorithm (ISA) biclustering algorithm as
implemented in the \code{isa2} R package. The clusters are "soft" meaning
that a story can appear in multiple clusters.

Install \code{isa2} package by running the command
\code{install.packages(\"isa2\")} before calling this function.
}
\examples{
\dontrun{
# Cluster "The Twilight Zone" franchise stories according to thematic
# similarity:
library(dplyr)
set_lto("demo")
set.seed(123)
result_tbl <- get_story_clusters()
result_tbl

# Explore a cluster of stories related to traveling back in time:
cluster_id <- 3
pull(result_tbl, stories)[[cluster_id]]
pull(result_tbl, themes)[[cluster_id]]

# Explore a cluster of stories related to mass panics:
cluster_id <- 5
pull(result_tbl, stories)[[cluster_id]]
pull(result_tbl, themes)[[cluster_id]]

# Explore a cluster of stories related to executions:
cluster_id <- 7
pull(result_tbl, stories)[[cluster_id]]
pull(result_tbl, themes)[[cluster_id]]

# Explore a cluster of stories related to space aliens:
cluster_id <- 10
pull(result_tbl, stories)[[cluster_id]]
pull(result_tbl, themes)[[cluster_id]]

# Explore a cluster of stories related to old people wanting to be young:
cluster_id <- 11
pull(result_tbl, stories)[[cluster_id]]
pull(result_tbl, themes)[[cluster_id]]

# Explore a cluster of stories related to wish making:
cluster_id <- 13
pull(result_tbl, stories)[[cluster_id]]
pull(result_tbl, themes)[[cluster_id]]
}
}
\references{
Gábor Csárdi, Zoltán Kutalik, Sven Bergmann (2010). Modular analysis of
gene expression data with R. \emph{Bioinformatics}, 26, 1376-7.

Sven Bergmann, Jan Ihmels, Naama Barkai (2003). Iterative signature
algorithm for the analysis of large-scale gene expression data.
\emph{Physical Review E}, 67, 031902.

Gábor Csárdi (2017). isa2: The Iterative Signature Algorithm. R package
version 0.3.5. \url{https://cran.r-project.org/package=isa2}
}
