use_fxns_urls <- list(
  tropicos = "http://services.tropicos.org/help?requestkey",
  entrez = "https://www.ncbi.nlm.nih.gov/account/"
)

#' Key helpers
#'
#' @importFrom utils browseURL
#' @title Helpers to set up authentication for the different providers.
#'
#' @description Sets up authentication to diverse providers by providing
#' the user a detailed prompt.
#'
#' @name key_helpers
#' @seealso [taxize-authentication]
NULL

#' @section `use_tropicos()`:
#' Browses to Tropicos API key request URL and provides instruction on how to
#' store the key. After filling the form you will get the key soon, but
#' not immediately.
#' @export
#' @rdname key_helpers
use_tropicos <- function() {
  if (interactive()) {
    utils::browseURL(use_fxns_urls$tropicos)
  }

  message(paste0(
    "After getting your key set it as TROPICOS_KEY in .Renviron.\n ",
    "TROPICOS_KEY='youractualkeynotthisstring'\n ",
    "For that, use usethis::edit_r_environ()"))

  invisible(use_fxns_urls$tropicos)
}

#' use eol
#' @export
#' @name use_eol-defunct
#' @keywords internal
use_eol <- function() {
  .Defunct(msg = "EOL no longer requires an API key")
}

#' @section `use_entrez()`:
#' Browse NCBI Entrez to help make an API key request and provides instruction
#' on how to store the key. There's no direct URL to request a key, one first
#' needs to log in or register and then to generate a key from one's account.
#'
#' Note that NCBI Entrez doesn't require that you use an API key,
#' but you should get higher rate limit with a key, so do get one.
#' @export
#' @rdname key_helpers
use_entrez <- function() {
  if (interactive()) {
    utils::browseURL(use_fxns_urls$entrez)
  }

  message(paste0(
    "Create your key from your (brand-new) account's. \n ",
    "After generating your key set it as ENTREZ_KEY in .Renviron.\n ",
    "ENTREZ_KEY='youractualkeynotthisstring'\n ",
    "For that, use usethis::edit_r_environ()"))

  invisible(use_fxns_urls$entrez)
}

#' @section `use_iucn()`:
#' Browse IUCN Red List API key request URL and provides instruction on how
#' to store the key. This function wraps [rredlist::rl_use_iucn()]
#' from the `rredlist` package. After filling the form you will get
#' the key soon, but not immediately.
#' @export
#' @rdname key_helpers
use_iucn <- function() {
  rredlist::rl_use_iucn()
}
