% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tin.R
\name{tin}
\alias{tin}
\alias{tin.character}
\alias{tin.matrix}
\alias{tin.list}
\alias{print.t2d_tin}
\title{TIN object}
\usage{
tin(x, ...)

\method{tin}{character}(x, ...)

\method{tin}{matrix}(x, ..., ikle, ipobo)

\method{tin}{list}(x, ..., s, s_brk, a, q = 30)

\method{print}{t2d_tin}(x, ...)
}
\arguments{
\item{x}{Either: a \code{character} string providing the name of a SELAFIN
file of which the mesh will be extracted;
a \code{matrix} of mesh points with 2 columns containing the x and y coordinates
(arguments \code{ikle} and \code{ipobo} are required);
a \code{list} with boundary and breakline definitions (see \code{Details}).}

\item{...}{Arguments passed to or from other methods. If \code{x} is a \code{list},
further arguments passed to \code{\link[RTriangle]{triangulate}}.}

\item{ikle}{If \code{x} is a matrix of points: A \code{matrix} with 3 columns
of indices referring to rows in \code{x}; each row represents a mesh
element (triangle). In TELEMAC termed \code{IKLE}.}

\item{ipobo}{If \code{x} is a matrix of points: A \code{vector} of indices referring
to rows in \code{x}, each marking a boundary point. In TELEMAC termed \code{IPOBO}.}

\item{s}{\code{numeric}, if \code{x} is a \code{list}: OPTIONAL value giving the
resolution of vertices along the boundary line for triangulation. If not given,
the points are used as they are supplied, otherwise \code{line_spacing} is called to ensure
equal spacing of vertices with the given segment lengths.}

\item{s_brk}{As \code{s} but for breaklines.}

\item{a}{\code{numeric}, maximum triangle area; passed to \code{\link[RTriangle]{triangulate}}.
Default: squared spacing of points (either given as \code{s} or inferred from \code{x$boundary}).}

\item{q}{\code{numeric}, minimum triangle angle; passed to \code{\link[RTriangle]{triangulate}}.
Default: 30 degrees.}
}
\value{
An object of class \code{t2d_tin}, which is a list with the following elements:
\describe{
  \item{points}{A \code{matrix} with the x and y coordinates (as columns) of mesh points.}
  \item{triangles}{A \code{matrix} with 3 columns of indices referring to rows in
    \code{points}; each row represents a mesh element (triangle).}
  \item{edges}{A \code{matrix} with 2 columns of indices referring to rows in
    \code{points}, the node points; each row represents an edge / segment of a triangle.}
  \item{boundaries}{A \code{vector} of indices referring to rows in \code{points}, each
    marking a point of the outer catchment boundary.}
  \item{breaklines}{A \code{matrix} with 2 columns of indices referring to rows in
    \code{points}, the vertices of the breaklines (used for mesh refinement during
    triangulation).}
}
}
\description{
Initialise a TIN mesh object for use within TELEMAC.
}
\details{
If \code{x} is a \code{list} this function creates a Triangulated Irregular Network
(TIN) using function \code{\link[RTriangle]{triangulate}}. The following \code{list}
elements are required to perform the triangulation:
\describe{
  \item{boundary}{A \code{matrix}, \code{data.frame}, \code{SpatialLines*} or \code{sf} object
    with two columns, each row defining a point along
    the outer catchment boundary. Points are connected one-by-one to a line starting
    with the first point, i.e. make sure points are in the right order! The first and
    last point will be connected to close the boundary.}
  \item{breaklines}{OPTIONAL, a \code{matrix}, \code{data.frame}, \code{SpatialLines*} or \code{sf}
    object with three columns
    \code{x} and \code{y}, the x and y coordinates of vertices along the breaklines,
    and \code{line}, an identifier to identify individual breaklines.}
}
}
\note{
Duplicated mesh points are silently removed.

Make sure breaklines do not intersect as this is not supported by the Triangle
algorithm. A possible workaround to split intersecting breaklines in R using
[sf](https://r-spatial.github.io/sf) is shown in the examples.

If you want to construct a \code{t2d_tin} object and get the error
\code{Boundary points do not comply with requirements: [...]}
the reason might be that breaklines are too close to the boundary causing that
points of the breaklines are used as boundary points which eventually results
in a discontinuous outer boundary. Try to increase the distance of breaklines
to the catchment boundary.
}
\examples{

### BASIC FUNCTIONALITY ###
library(sf)

# load boundary as sf linestring
bnd <- st_read(system.file("dem/boundary_lagos.gpkg", package = "telemac"))

# create t2d_tin object
tin_obj <- tin(list(boundary = bnd), s = 90, a = 100^2, q = 30)

# inspection
tin_obj
str(tin_obj)
plot(tin_obj, pch = ".")

### DEALING WITH INTERSECTING BREAKLINES ###
library(sf)
library(tidyverse)

# example boundary and with intersecting breaklines
test_bnd <- st_linestring(
  matrix(c(seq(0,100,5),  rep(100,21), seq(100,0,-5),     rep(0,21),
           rep(0,21), seq(0,100,5),   rep(100,21), seq(100,0,-5)),
         ncol = 2)
) \%>\% st_sfc()
test_brk <- list(
  st_linestring(matrix(c(seq(0,100,5), rep(50,21)), ncol = 2)),
  st_linestring(matrix(c(rep(50,21), seq(0,100,5)), ncol = 2)),
  st_linestring(matrix(c(seq(30,60,5), rep(60,11),
                         rep(20,7), seq(20,70,5)), ncol = 2))) \%>\% st_sfc()

# get intersection points and define buffer of 2 around these points
pt_inters <- c(test_bnd, test_brk) \%>\%
  st_intersection() \%>\%
  st_collection_extract(type = "POINT") \%>\%
  st_buffer(2)

plot(test_bnd)
plot(test_brk, add = TRUE)
plot(pt_inters, add = TRUE)

# split breaklines
test_brk_unique <- st_difference(st_union(test_brk), st_union(pt_inters))
plot(test_bnd)
plot(test_brk_unique, add = TRUE)

# create mesh
tin_obj <- tin(list(boundary = test_bnd, breaklines = test_brk_unique),
               s = 2, s_brk = 2, a = 4, q = 30)
plot(tin_obj, pch = ".")
}
