%  File man/tergm.godfather.Rd in package tergm, part of the Statnet suite
%  of packages for network analysis, http://statnet.org .
%
%  This software is distributed under the GPL-3 license.  It is free,
%  open source, and has the attribution requirements (GPL Section 7) at
%  http://statnet.org/attribution
%
%  Copyright 2003-2013 Statnet Commons
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
\name{tergm.godfather}
\alias{tergm.godfather}
\alias{control.tergm.godfather}
\title{
  A function to apply a given series of changes a network.
}
\description{
  Gives the network a series of timed proposals it can't refuse. Returns the
  statistics of the network, and, optionally, the final network.
}
\usage{
tergm.godfather(formula, changes = NULL, toggles = changes[, -4, drop = FALSE],
  start = NULL, end = NULL, end.network = FALSE, stats.start = FALSE, verbose = FALSE,
  control = control.tergm.godfather())
}
\arguments{
  \item{formula}{
    An \code{\link{summary.formula}}-style formula, with either a
    \code{\link{network}} or a \code{\link{networkDynamic}} as the LHS
    and statistics to be computed on the RHS. If LHS is a
    \code{\link{networkDynamic}}, it will be used to derive the changes
    to the network whose statistics are wanted. Otherwise, either \code{changes}
    or \code{toggles} must be specified, and the LHS
    \code{\link{network}} will be used as the starting network.
  }
  \item{changes}{
    A matrix with four columns: time, tail, head, and new value,
    describing the changes to be made. Can only be used if LHS of
    \code{formula} is not a \code{\link{networkDynamic}}.
  }
  \item{toggles}{
    A matrix with three columns: time, tail, and head, giving the dyads
    which had changed. Can only be used if LHS of
    \code{formula} is not a \code{\link{networkDynamic}}.
  }
  \item{start}{
    Time from which to start applying changes. Note that the first
    set of changes will take effect at \code{start+1}. Defaults to the
    time point 1 before the earliest change passed.
  }
  \item{end}{
    Time from which to finish applying changes. Defaults to the
    last time point at which a change occurs.
  }
  \item{end.network}{
    Whether to return a network that results. Defaults to \code{FALSE}.
  }
  \item{stats.start}{
    Whether to return the network statistics at \code{start} (before any
    changes are applied) as the first row of the statistics
    matrix. Defaults to \code{FALSE}, to produce output similar to that
    of \code{\link[=simulate.stergm]{simulate}} for STERGMs when
    \code{output="stats"}, where initial network's statistics are not
    returned.
  }
  \item{verbose}{
    Whether to print progress messages.
  }
  \item{control}{
    A control list generated by
    \code{\link{control.tergm.godfather}}. So far, there is only one
    argument: \code{GF.init.maxedges.mul}, to control how much space is
    allocated for the edgelist of the final network. It is used
    adaptively, so should not be greater than \code{10}.
  }
}
\value{
  If \code{end.network==FALSE} (the default), an \code{\link{mcmc}}
  object with the requested network statistics associed with the network series
  produced by applying the specified changes. Its \code{\link{mcmc}}
  attributes encode the timing information: so \code{\link{start}(out)}
  gives the time point associated with the first row returned, and
  \code{\link{end}(out)} out the last. The "thinning interval" is always 1.

  If \code{end.network==TRUE}, return a \code{\link{network}} object
  with \code{\link{lasttoggle}} "extension", representing the final
  network, with a matrix of statistics described in the previous paragraph attached to it as an \code{attr}-style
  attribute \code{"stats"}.
}
\seealso{
  simulate.stergm, simulate.network, simulate.networkDynamic
}
\examples{

g1 <- network.initialize(10, dir=FALSE)
g1[1,2] <- 1
g1[3,4] <- 1
g1 \%n\% "time" <- 0
g1 \%n\% "lasttoggle" <- -1-rgeom(network.dyadcount(g1),1/4)

dc <- matrix(rnorm(100),10,10); dc <- dc+t(dc)

# Simulate a network, tracking its statistics.
simnet <- simulate(g1, formation=~edges, dissolution=~edges, coef.form=-1, coef.diss=1,
                   time.slices=50, monitor=~degree(1)+mean.age+degree.mean.age(1)+
                                            edge.ages+edgecov(dc)+edgecov.ages(dc),
                   output="networkDynamic")

sim.stats <- attr(simnet, "stats")

print(head(sim.stats))
sim.stats <- as.matrix(sim.stats)

# Replay the simulation using a networkDynamic, monitoring a potentially different set of
# statistics (but same in this case).
gf1.stats <- tergm.godfather(simnet~degree(1)+mean.age+degree.mean.age(1)+
                                    edge.ages+edgecov(dc)+edgecov.ages(dc),
                             start=0, end=50)

print(head(gf1.stats))
gf1.stats <- as.matrix(gf1.stats)

# Replay the simulation using the initial network + list of changes.

gf2.stats <- tergm.godfather(g1~degree(1)+mean.age+degree.mean.age(1)+
                                edge.ages+edgecov(dc)+edgecov.ages(dc),
                             start=0, end=50, changes=attr(simnet,"changes"))

print(head(gf2.stats))
gf2.stats <- as.matrix(gf2.stats)

# We can also compare them to the network statistics summarized.
summ.stats <- summary(simnet~degree(1)+mean.age+degree.mean.age(1)+
                             edge.ages+edgecov(dc)+edgecov.ages(dc), at=1:50)

print(head(summ.stats))

tol <- sqrt(.Machine$double.eps)
# If they aren't all identical, we are in trouble.
stopifnot(all.equal(sim.stats,gf1.stats),
          all.equal(sim.stats,gf2.stats),
          all.equal(sim.stats,summ.stats))

}
