% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tfb-fpc.R
\name{tfb_fpc}
\alias{tfb_fpc}
\alias{tfb_fpc.data.frame}
\alias{tfb_fpc.matrix}
\alias{tfb_fpc.numeric}
\alias{tfb_fpc.tf}
\alias{tfb_fpc.default}
\title{Functional data in FPC-basis representation}
\usage{
tfb_fpc(data, ...)

\method{tfb_fpc}{data.frame}(
  data,
  id = 1,
  arg = 2,
  value = 3,
  domain = NULL,
  method = fpc_wsvd,
  ...
)

\method{tfb_fpc}{matrix}(data, arg = NULL, domain = NULL, method = fpc_wsvd, ...)

\method{tfb_fpc}{numeric}(data, arg = NULL, domain = NULL, method = fpc_wsvd, ...)

\method{tfb_fpc}{tf}(data, arg = NULL, method = fpc_wsvd, ...)

\method{tfb_fpc}{default}(data, arg = NULL, domain = NULL, method = fpc_wsvd, ...)
}
\arguments{
\item{data}{a \code{matrix}, \code{data.frame} or \code{list} of suitable shape, or another
\code{tf}-object containing functional data.}

\item{...}{arguments to the \code{method} which computes the
(regularized/smoothed) FPCA - see e.g. \code{\link[=fpc_wsvd]{fpc_wsvd()}}.
Unless set by the user, uses proportion of variance explained
\code{pve = 0.995} to determine the truncation levels.}

\item{id}{The name or number of the column defining which data belong to
which function.}

\item{arg}{\code{numeric}, or list of \code{numeric}s. The evaluation grid.
For the \code{data.frame}-method: the
name/number of the column defining the evaluation grid. The \code{matrix} method
will try to guess suitable \code{arg}-values from the column names of \code{data} if
\code{arg} is not supplied. Other methods fall back on integer sequences
(\verb{1:<length of data>}) as the default if not provided.}

\item{value}{The name or number of the column containing the function
evaluations.}

\item{domain}{range of the \code{arg}.}

\item{method}{the function to use that computes eigenfunctions and scores.
Defaults to \code{\link[=fpc_wsvd]{fpc_wsvd()}}, which is quick and easy but returns completely
unsmoothed eigenfunctions unlikely to be suited for noisy data. See Details.}
}
\value{
an object of class \code{tfb_fpc}, inheriting from \code{tfb}.
The basis used by \code{tfb_fpc} is a \code{tfd}-vector containing the estimated
mean and eigenfunctions.
}
\description{
These functions perform a (functional) principal component analysis (FPCA) of
the input data and return an \code{tfb_fpc} \code{tf}-object that uses the empirical
eigenfunctions as basis functions for representing the data. The default
("\code{method = fpc_wsvd}") uses a (truncated) weighted SVD for complete
data on a common grid and a nuclear-norm regularized (truncated) weighted SVD
for partially missing data on a common grid, see \code{\link[=fpc_wsvd]{fpc_wsvd()}}.
The latter is likely to break down for high PVE and/or high amounts of
missingness.\cr
}
\details{
For the FPC basis, any factorization method that accepts a \code{data.frame} with
columns \code{id}, \code{arg}, \code{value} containing the functional data and returns a
list with eigenfunctions and FPC scores structured like the return object
of \code{\link[=fpc_wsvd]{fpc_wsvd()}} can be used for the `method`` argument, see example below.
Note that the mean function, with a fixed "score" of 1 for all functions,
is used as the first basis function for all FPC bases.
}
\section{Methods (by class)}{
\itemize{
\item \code{tfb_fpc(default)}: convert \code{tfb}: default method, returning prototype when
data is NULL

}}
\examples{
set.seed(13121)
x <- tf_rgp(25, nugget = .02)
x_pc <- tfb_fpc(x, pve = .9)
x_pc
plot(x, lwd = 3)
lines(x_pc, col = 2, lty = 2)
x_pc_full <- tfb_fpc(x, pve = .995)
x_pc_full
lines(x_pc_full, col = 3, lty = 2)
# partially missing data on common grid:
x_mis <- x |> tf_sparsify(dropout = .05)
x_pc_mis <- tfb_fpc(x_mis, pve = .9)
x_pc_mis
plot(x_mis, lwd = 3)
lines(x_pc_mis, col = 4, lty = 2)
# extract FPC basis --
# first "eigenvector" in black is (always) the mean function
x_pc |> tf_basis(as_tfd = TRUE) |> plot(col = 1:5)
\donttest{
# Apply FPCA for sparse, irregular data using refund::fpca.sc:
set.seed(99290)
# create small, sparse, irregular data:
x_irreg <- x[1:8] |>
  tf_jiggle() |> tf_sparsify(dropout = 0.3)
plot(x_irreg)
x_df <- x_irreg |>
  as.data.frame(unnest = TRUE)
# wrap refund::fpca_sc for use as FPCA method in tfb_fpc --
# 1. define scoring function (simple weighted LS fit)
fpca_scores <- function(data_matrix, efunctions, mean, weights) {
  w_mat <- matrix(weights, ncol = length(weights), nrow = nrow(data_matrix),
                  byrow = TRUE)
  w_mat[is.na(data_matrix)] <- 0
  data_matrix[is.na(data_matrix)] <- 0
  data_wc <- t((t(data_matrix) - mean) * sqrt(t(w_mat)))
  t(qr.coef(qr(efunctions), t(data_wc) / sqrt(weights)))
}
# 2. define wrapper for fpca_sc:
fpca_sc_wrapper <- function(data, arg, pve = 0.995, ...) {
  data_mat <- tfd(data) |> as.matrix(interpolate = TRUE)
  fpca <- refund::fpca.sc(
    Y = data_mat, argvals = attr(data_mat, "arg"), pve = pve, ...
  )
  c(fpca[c("mu", "efunctions", "scores", "npc")],
    scoring_function = fpca_scores)
}
x_pc <- tfb_fpc(x_df, method = fpca_sc_wrapper)
lines(x_pc, col = 2, lty = 2)
}
}
\seealso{
\code{\link[=fpc_wsvd]{fpc_wsvd()}} for FPCA options.

Other tfb-class: 
\code{\link{fpc_wsvd}()},
\code{\link{tfb}},
\code{\link{tfb_spline}()}

Other tfb_fpc-class: 
\code{\link{fpc_wsvd}()}
}
\concept{tfb-class}
\concept{tfb_fpc-class}
