% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nn-conv.R
\name{nn_conv2d}
\alias{nn_conv2d}
\title{Conv2D module}
\usage{
nn_conv2d(
  in_channels,
  out_channels,
  kernel_size,
  stride = 1,
  padding = 0,
  dilation = 1,
  groups = 1,
  bias = TRUE,
  padding_mode = "zeros"
)
}
\arguments{
\item{in_channels}{(int): Number of channels in the input image}

\item{out_channels}{(int): Number of channels produced by the convolution}

\item{kernel_size}{(int or tuple): Size of the convolving kernel}

\item{stride}{(int or tuple, optional): Stride of the convolution. Default: 1}

\item{padding}{(int or tuple, optional): Zero-padding added to both sides of
the input. Default: 0}

\item{dilation}{(int or tuple, optional): Spacing between kernel elements. Default: 1}

\item{groups}{(int, optional): Number of blocked connections from input
channels to output channels. Default: 1}

\item{bias}{(bool, optional): If \code{TRUE}, adds a learnable bias to the
output. Default: \code{TRUE}}

\item{padding_mode}{(string, optional): \code{'zeros'}, \code{'reflect'},
\code{'replicate'} or \code{'circular'}. Default: \code{'zeros'}}
}
\description{
Applies a 2D convolution over an input signal composed of several input
planes.
}
\details{
In the simplest case, the output value of the layer with input size
\eqn{(N, C_{\mbox{in}}, H, W)} and output \eqn{(N, C_{\mbox{out}}, H_{\mbox{out}}, W_{\mbox{out}})}
can be precisely described as:

\deqn{
\mbox{out}(N_i, C_{\mbox{out}_j}) = \mbox{bias}(C_{\mbox{out}_j}) +
  \sum_{k = 0}^{C_{\mbox{in}} - 1} \mbox{weight}(C_{\mbox{out}_j}, k) \star \mbox{input}(N_i, k)
}

where \eqn{\star} is the valid 2D cross-correlation operator,
\eqn{N} is a batch size, \eqn{C} denotes a number of channels,
\eqn{H} is a height of input planes in pixels, and \eqn{W} is
width in pixels.
\itemize{
\item \code{stride} controls the stride for the cross-correlation, a single
number or a tuple.
\item \code{padding} controls the amount of implicit zero-paddings on both
sides for \code{padding} number of points for each dimension.
\item \code{dilation} controls the spacing between the kernel points; also
known as the à trous algorithm. It is harder to describe, but this \code{link}_
has a nice visualization of what \code{dilation} does.
\item \code{groups} controls the connections between inputs and outputs.
\code{in_channels} and \code{out_channels} must both be divisible by
\code{groups}. For example,
\itemize{
\item At groups=1, all inputs are convolved to all outputs.
\item At groups=2, the operation becomes equivalent to having two conv
layers side by side, each seeing half the input channels,
and producing half the output channels, and both subsequently
concatenated.
\item At groups= \code{in_channels}, each input channel is convolved with
its own set of filters, of size:
\eqn{\left\lfloor\frac{out\_channels}{in\_channels}\right\rfloor}.
}
}

The parameters \code{kernel_size}, \code{stride}, \code{padding}, \code{dilation} can either be:
\itemize{
\item a single \code{int} -- in which case the same value is used for the height and
width dimension
\item a \code{tuple} of two ints -- in which case, the first \code{int} is used for the height dimension,
and the second \code{int} for the width dimension
}
}
\section{Note}{


Depending of the size of your kernel, several (of the last)
columns of the input might be lost, because it is a valid cross-correlation,
and not a full cross-correlation.
It is up to the user to add proper padding.

When \code{groups == in_channels} and \code{out_channels == K * in_channels},
where \code{K} is a positive integer, this operation is also termed in
literature as depthwise convolution.
In other words, for an input of size :math:\verb{(N, C_\{in\}, H_\{in\}, W_\{in\})},
a depthwise convolution with a depthwise multiplier \code{K}, can be constructed by arguments
\eqn{(in\_channels=C_{in}, out\_channels=C_{in} \times K, ..., groups=C_{in})}.

In some circumstances when using the CUDA backend with CuDNN, this operator
may select a nondeterministic algorithm to increase performance. If this is
undesirable, you can try to make the operation deterministic (potentially at
a performance cost) by setting \code{backends_cudnn_deterministic = TRUE}.
}

\section{Shape}{

\itemize{
\item Input: \eqn{(N, C_{in}, H_{in}, W_{in})}
\item Output: \eqn{(N, C_{out}, H_{out}, W_{out})} where
\deqn{
  H_{out} = \left\lfloor\frac{H_{in}  + 2 \times \mbox{padding}[0] - \mbox{dilation}[0]
    \times (\mbox{kernel\_size}[0] - 1) - 1}{\mbox{stride}[0]} + 1\right\rfloor
}
\deqn{
  W_{out} = \left\lfloor\frac{W_{in}  + 2 \times \mbox{padding}[1] - \mbox{dilation}[1]
    \times (\mbox{kernel\_size}[1] - 1) - 1}{\mbox{stride}[1]} + 1\right\rfloor
}
}
}

\section{Attributes}{

\itemize{
\item weight (Tensor): the learnable weights of the module of shape
\eqn{(\mbox{out\_channels}, \frac{\mbox{in\_channels}}{\mbox{groups}}},
\eqn{\mbox{kernel\_size[0]}, \mbox{kernel\_size[1]})}.
The values of these weights are sampled from
\eqn{\mathcal{U}(-\sqrt{k}, \sqrt{k})} where
\eqn{k = \frac{groups}{C_{\mbox{in}} * \prod_{i=0}^{1}\mbox{kernel\_size}[i]}}
\item bias (Tensor): the learnable bias of the module of shape
(out_channels). If \code{bias} is \code{TRUE},
then the values of these weights are
sampled from \eqn{\mathcal{U}(-\sqrt{k}, \sqrt{k})} where
\eqn{k = \frac{groups}{C_{\mbox{in}} * \prod_{i=0}^{1}\mbox{kernel\_size}[i]}}
}
}

\examples{
if (torch_is_installed()) {

# With square kernels and equal stride
m <- nn_conv2d(16, 33, 3, stride = 2)
# non-square kernels and unequal stride and with padding
m <- nn_conv2d(16, 33, c(3, 5), stride=c(2, 1), padding=c(4, 2))
# non-square kernels and unequal stride and with padding and dilation
m <- nn_conv2d(16, 33, c(3, 5), stride=c(2, 1), padding=c(4, 2), dilation=c(3, 1))
input <- torch_randn(20, 16, 50, 100)
output <- m(input)  

}
}
