% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nlp_scores.R
\name{nlp_scores}
\alias{nlp_scores}
\title{Natural Language Processing Scores}
\usage{
nlp_scores(
  text,
  classes,
  semantic_space = c("baroni", "cbow", "cbow_ukwac", "en100", "glove", "tasa"),
  preprocess = TRUE,
  remove_stop = TRUE,
  keep_in_env = TRUE,
  envir = 1
)
}
\arguments{
\item{text}{Character vector or list.
Text in a vector or list data format}

\item{classes}{Character vector.
Classes to score the text}

\item{semantic_space}{Character vector.
The semantic space used to compute the distances between words
(more than one allowed). Here's a list of the semantic spaces:

\itemize{

\item{\code{"baroni"}}
{Combination of British National Corpus, ukWaC corpus, and a 2009
Wikipedia dump. Space created using continuous bag of words algorithm
using a context window size of 11 words (5 left and right)
and 400 dimensions. Best word2vec model according to
Baroni, Dinu, & Kruszewski (2014)}

\item{\code{"cbow"}}
{Combination of British National Corpus, ukWaC corpus, and a 2009
Wikipedia dump. Space created using continuous bag of words algorithm with
a context window size of 5 (2 left and right) and 300 dimensions}

\item{\code{"cbow_ukwac"}}
{ukWaC corpus with the continuous bag of words algorithm with
a context window size of 5 (2 left and right) and 400 dimensions}

\item{\code{"en100"}}
{Combination of British National Corpus, ukWaC corpus, and a 2009
Wikipedia dump. 100,000 most frequent words. Uses moving window model
with a size of 5 (2 to the left and right). Positive pointwise mutual
information and singular value decomposition was used to reduce the
space to 300 dimensions}

\item{\code{"glove"}}
{\href{https://dumps.wikimedia.org/}{Wikipedia 2014 dump} and \href{https://catalog.ldc.upenn.edu/LDC2011T07}{Gigaword 5} with 400,000
words (300 dimensions). Uses co-occurrence of words in text
documents (uses cosine similarity)}

\item{\code{"tasa"}}
{Latent Semantic Analysis space from TASA corpus all (300 dimensions).
Uses co-occurrence of words in text documents (uses cosine similarity)}

}}

\item{preprocess}{Boolean.
Should basic preprocessing be applied?
Includes making lowercase, keeping only alphanumeric characters,
removing escape characters, removing repeated characters,
and removing white space.
Defaults to \code{TRUE}}

\item{remove_stop}{Boolean.
Should \code{\link[transforEmotion]{stop_words}}
be removed?
Defaults to \code{TRUE}}

\item{keep_in_env}{Boolean.
Whether the classifier should be kept in your global environment.
Defaults to \code{TRUE}.
By keeping the classifier in your environment, you can skip
re-loading the classifier every time you run this function.
\code{TRUE} is recommended}

\item{envir}{Numeric.
Environment for the classifier to be saved for repeated use.
Defaults to the global environment}
}
\value{
Returns semantic distances for the text classes
}
\description{
Natural Language Processing using word embeddings to compute
semantic similarities (cosine) of text and specified classes
}
\examples{
# Load data
data(neo_ipip_extraversion)

# Example text 
text <- neo_ipip_extraversion$friendliness[1:5]

\dontrun{
# GloVe
nlp_scores(
 text = text,
 classes = c(
   "friendly", "gregarious", "assertive",
   "active", "excitement", "cheerful"
 )
)

# Baroni
nlp_scores(
 text = text,
 classes = c(
   "friendly", "gregarious", "assertive",
   "active", "excitement", "cheerful"
 ),
 semantic_space = "baroni"
)
 
# CBOW
nlp_scores(
 text = text,
 classes = c(
   "friendly", "gregarious", "assertive",
   "active", "excitement", "cheerful"
 ),
 semantic_space = "cbow"
)

# CBOW + ukWaC
nlp_scores(
 text = text,
 classes = c(
   "friendly", "gregarious", "assertive",
   "active", "excitement", "cheerful"
 ),
 semantic_space = "cbow_ukwac"
)

# en100
nlp_scores(
 text = text,
 classes = c(
   "friendly", "gregarious", "assertive",
   "active", "excitement", "cheerful"
 ),
 semantic_space = "en100"
)

# tasa
nlp_scores(
 text = text,
 classes = c(
   "friendly", "gregarious", "assertive",
   "active", "excitement", "cheerful"
 ),
 semantic_space = "tasa"
)
}

}
\references{
Baroni, M., Dinu, G., & Kruszewski, G. (2014).
Don't count, predict! a systematic comparison of context-counting vs. context-predicting semantic vectors.
In \emph{Proceedings of the 52nd annual meting of the association for computational linguistics} (pp. 238-247).

Landauer, T.K., & Dumais, S.T. (1997).
A solution to Plato's problem: The Latent Semantic Analysis theory of acquisition, induction and representation of knowledge.
\emph{Psychological Review}, \emph{104}, 211-240.

Pennington, J., Socher, R., & Manning, C. D. (2014).
GloVe: Global vectors for word representation.
In \emph{Proceedings of the 2014 conference on empirical methods in natural language processing} (pp. 1532-1543).
}
\author{
Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
