#' Visual Studio Project Repositories
#'
#' @description
#' These functions will allow you to scrape project information from Visual Studio.
#'
#' @details
#' For more information about repository API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/git/}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param repo the name of the repository in \code{project} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param quiet logical whether want general running information from printing. Any issue with the API call will
#' still show up if set to \code{TRUE}
#'
#' @examples
#' #Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key('<username>', '<password>')
#'
#' #Get repo list
#' vsts_get_repos('domain', 'project', auth_key)
#'
#' #Create new repo
#' vsts_create_repo('domain', 'project', 'repo', auth_key)
#'
#' #Delete existing repo
#' vsts_delete_repo('domain', 'project', 'repo', auth_key)
#'
#'
#' @rdname vsts_repo
#' @export
vsts_get_repos <- function(domain, project, auth_key, quiet = FALSE) {
  uri <- paste0('https://', domain, '.visualstudio.com/', project, '/_apis/git/repositories?api-version=1.0')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key))
  if(httr::status_code(response) != 200) {
    cat(httr::http_condition(response, 'message', 'get repos list')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE) %>% .$value
  if(!quiet) cat('Available repositories:', paste(content$name, collapse = ', '), '\n')
  invisible(content)
}

#' @rdname vsts_repo
#' @export
vsts_create_repo <- function(domain, project, repo, auth_key, quiet = FALSE) {
  uri <- paste0('https://', domain, '.visualstudio.com/DefaultCollection/_apis/git/repositories?api-version=1.0')
  proj_id <- vsts_get_projects(domain, auth_key, quiet = TRUE) %>% .[.$name == project, 'id']
  if(is.null(proj_id)) return(invisible(NULL))
  if(length(proj_id) == 0) {
    cat('Unable to find', project, 'in', domain, '\n')
    return(invisible(NULL))
  }

  content_body <- jsonlite::toJSON(list(name = repo, project = list(id = proj_id)), auto_unbox = TRUE)

  response <- httr::POST(uri, httr::add_headers(Authorization = auth_key), httr::content_type_json(), body = content_body)
  if(httr::status_code(response) != 201) {
    if(httr::status_code(response) == 409) {
      fail_msg <- paste('create repository;', repo, 'already exists in', project)
    } else {
      fail_msg <- 'create repository'
    }
    cat(httr::http_condition(response, 'message', fail_msg)$message, '\n')
    return(invisible(NULL))
  }

  if(!quiet) cat(repo, 'repository has been created in', project, '\n')
  content <- data.frame(httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE))
  invisible(content)
}

#' @rdname vsts_repo
#' @export
vsts_delete_repo <- function(domain, project, repo, auth_key, quiet = FALSE) {
  repo_id <- vsts_get_repos(domain, project, auth_key, quiet = TRUE) %>% .[.$name == repo, 'id']
  if(length(repo_id) == 0) {
    cat('Unable to find', repo, 'in', project, '\n')
    return(invisible(NULL))
  }
  uri <- paste0('https://', domain, '.visualstudio.com/DefaultCollection/', project,
                '/_apis/git/repositories/', repo_id, '?api-version=1.0')

  response <- httr::DELETE(uri, httr::add_headers(Authorization = auth_key))
  if(response$status_code != 204) {
    cat('Unable to perform request, status code:', response$status_code, '\n')
    return(invisible(NULL))
  }

  if(!quiet) cat(repo, 'repository has been deleted from', project, '\n')
  return(invisible(TRUE))
}
