% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/walker_rw1.R
\name{walker_rw1}
\alias{walker_rw1}
\title{Comparison of naive and state space implementation of RW1 regression model}
\usage{
walker_rw1(formula, data, beta_prior, sigma_prior, init, chains,
  naive = FALSE, return_x_reg = FALSE, ...)
}
\arguments{
\item{formula}{An object of class \code{\link[stats]{formula}}. See \code{\link[stats]{lm}} for details.}

\item{data}{An optional data.frame or object coercible to such, as in \code{\link[stats]{lm}}.}

\item{beta_prior}{A matrix with \eqn{k} rows and 2 columns, where first columns defines the 
prior means of the Gaussian priors of the corresponding \eqn{k} regression coefficients, 
and the second column defines the the standard deviations of those prior distributions.}

\item{sigma_prior}{A matrix with \eqn{k + 1} rows and two colums with similar structure as 
\code{beta_prior}, with first row corresponding to the prior of the standard deviation of the 
observation level noise, and rest of the rows define the priors for the standard deviations of 
random walk noise terms. The prior distributions for all sigmas are 
Gaussians truncated to positive real axis. For non-Gaussian models, this should contain only k rows. 
For second order random walk model, these priors correspond to the slope level standard deviations.}

\item{init}{Initial value specification, see \code{\link[rstan]{sampling}}.}

\item{chains}{Number of Markov chains. Default is 4.}

\item{naive}{Only used for \code{walker} function. 
If \code{TRUE}, use "standard" approach which samples the joint posterior 
\eqn{p(beta, sigma | y)}. If \code{FALSE} (the default), use marginalisation approach 
where we sample the marginal posterior \eqn{p(sigma | y)} and generate the samples of 
\eqn{p(beta | sigma, y)} using state space modelling techniques 
(namely simulation smoother by Durbin and Koopman (2002)). Both methods give asymptotically 
identical results, but the latter approach is computationally much more efficient.}

\item{return_x_reg}{If \code{TRUE}, does not perform sampling, but instead returns the matrix of 
predictors after processing the \code{formula}.}

\item{...}{Additional arguments to \code{\link[rstan]{sampling}}.}
}
\description{
This function is the first iteration of the function \code{walker},
which supports only time-varying model where all coefficients ~ rw1.
This is kept as part of the package in order to compare "naive" and 
state space versions of the model in the vignette.
}
\examples{
\dontrun{
## Comparing the approaches, note that with such a small data 
## the differences aren't huge, but try same with n = 500 and/or more terms...
set.seed(123)
n <- 100
beta1 <- cumsum(c(0.5, rnorm(n - 1, 0, sd = 0.05)))
beta2 <- cumsum(c(-1, rnorm(n - 1, 0, sd = 0.15)))
x1 <- rnorm(n, 1)
x2 <- 0.25 * cos(1:n)
ts.plot(cbind(beta1 * x1, beta2 *x2), col = 1:2)
u <- cumsum(rnorm(n))
y <- rnorm(n, u + beta1 * x1 + beta2 * x2)
ts.plot(y)
lines(u + beta1 * x1 + beta2 * x2, col = 2)
kalman_walker <- walker_rw1(y ~ -1 + 
  rw1(~ x1 + x2, beta_prior = c(0, 2), sigma_prior = c(0, 2)), 
  sigma_y_prior = c(0, 2), iter = 2000, chains = 1)
print(kalman_walker$stanfit, pars = c("sigma_y", "sigma_rw1"))
betas <- extract(kalman_walker$stanfit, "beta")[[1]]
ts.plot(cbind(u, beta1, beta2, apply(betas, 2, colMeans)), 
  col = 1:3, lty = rep(2:1, each = 3))
sum(get_elapsed_time(kalman_walker$stanfit))
naive_walker <- walker_rw1(y ~ x1 + x2, iter = 2000, chains = 1, 
  beta_prior = cbind(0, rep(2, 3)), sigma_prior = cbind(0, rep(2, 4)), 
  naive = TRUE)
print(naive_walker$stanfit, pars = c("sigma_y", "sigma_b"))
sum(get_elapsed_time(naive_walker$stanfit))

## Larger problem, this takes some time with naive approach

set.seed(123)
n <- 500
beta1 <- cumsum(c(1.5, rnorm(n - 1, 0, sd = 0.05)))
beta2 <- cumsum(c(-1, rnorm(n - 1, 0, sd = 0.5)))
beta3 <- cumsum(c(-1.5, rnorm(n - 1, 0, sd = 0.15)))
beta4 <- 2
x1 <- rnorm(n, 1)
x2 <- 0.25 * cos(1:n)
x3 <- runif(n, 1, 3)
ts.plot(cbind(beta1 * x1, beta2 * x2, beta3 * x3), col = 1:3)
a <- cumsum(rnorm(n))
signal <- a + beta1 * x1 + beta2 * x2 + beta3 * x3
y <- rnorm(n, signal)
ts.plot(y)
lines(signal, col = 2)
kalman_walker <- walker_rw1(y ~ x1 + x2 + x3, iter = 2000, chains = 1,
  beta_prior = cbind(0, rep(2, 4)), sigma_prior = cbind(0, rep(2, 5)))
print(kalman_walker$stanfit, pars = c("sigma_y", "sigma_b"))
betas <- extract(kalman_walker$stanfit, "beta")[[1]]
ts.plot(cbind(u, beta1, beta2, beta3, apply(betas, 2, colMeans)), 
  col = 1:4, lty = rep(2:1, each = 4))
sum(get_elapsed_time(kalman_walker$stanfit))
# need to increase adapt_delta in order to get rid of divergences
# and max_treedepth to get rid of related warnings
# and still we end up with low BFMI warning after hours of computation
naive_walker <- walker_rw1(y ~ x1 + x2 + x3, iter = 2000, chains = 1, 
  beta_prior = cbind(0, rep(2, 4)), sigma_prior = cbind(0, rep(2, 5)),
  naive = TRUE, control = list(adapt_delta = 0.9, max_treedepth = 15)) 
print(naive_walker$stanfit, pars = c("sigma_y", "sigma_b"))
sum(get_elapsed_time(naive_walker$stanfit))
}

}
