#' Weight matrix
#'
#' This function compiles a weight matrix according to one of several weighting
#' schemas and allows users to visualize the impact of the weight matrix on each
#' element of the confusion matrix.
#'
#' @param n the number of classes contained in the confusion matrix.
#'
#' @param weight.type the weighting schema to be used. Can be one of:
#' "arithmetic" - a decreasing arithmetic progression weighting scheme,
#' "geometric" - a decreasing geometric progression weighting scheme,
#' "normal" - weights drawn from the right tail of a normal distribution,
#' "interval" - weights contained on a user-defined interval,
#' "custom" - custom weight vector defined by the user.
#'
#' @param weight.penalty determines whether the weights associated with
#' non-diagonal elements generated by the "normal", "arithmetic" and "geometric"
#' weight types are positive or negative values. By default, the value is set to
#' FALSE, which means that generated weights will be positive values.
#'
#' @param standard.deviation standard deviation of the normal distribution, if
#' the normal distribution weighting schema is used.
#'
#' @param geometric.multiplier the multiplier used to construct the geometric
#' progression series, if the geometric progression weighting scheme is used.
#'
#' @param interval.high the upper bound of the weight interval, if the interval
#' weighting scheme is used.
#'
#' @param interval.low the lower bound of the weight interval, if the interval
#' weighting scheme is used.
#'
#' @param custom.weights the vector of custom weights to be applied, is the
#' custom weighting scheme was selected. The vector should be equal to "n", but
#' can be larger, with excess values being ignored.
#'
#' @param plot.weights optional setting to enable plotting of weight vector,
#' corresponding to the first column of the weight matrix
#'
#' @return an nxn matrix, containing the weights to be multiplied with the
#'   confusion matrix.
#'
#' @details The number of categories "n" should be greater or equal to 2.
#'
#' @usage weightmatrix(n, weight.type = "arithmetic", weight.penalty = FALSE,
#'                     standard.deviation = 2,
#'                     geometric.multiplier = 2,
#'                     interval.high=1, interval.low = -1,
#'                     custom.weights = NA,
#'                     plot.weights = FALSE)
#'
#' @keywords weight matrix normal arithmetic geometric progression interval
#'
#' @seealso [wconfusionmatrix()]
#'
#' @author Alexandru Monahov, <https://www.alexandrumonahov.eu.org/>
#'
#' @examples
#' weightmatrix(n=4, weight.type="arithmetic", plot.weights = TRUE)
#' weightmatrix(n=4, weight.type="normal", standard.deviation = 1,
#'                   plot.weights = TRUE)
#' weightmatrix(n=4, weight.type="interval", interval.high = 1,
#'                   interval.low = -0.5, plot.weights = TRUE)
#' weightmatrix(n=4, weight.type="geometric", geometric.multiplier = 0.6)
#' weightmatrix(n=4, weight.type="custom", custom.weights = c(1,0.2,0.1,0),
#'                   plot.weights = TRUE)
#'
#' @export

weightmatrix <- function(n, weight.type = "arithmetic", weight.penalty = FALSE, standard.deviation = 2, geometric.multiplier = 2, interval.high=1, interval.low = -1, custom.weights = NA, plot.weights = FALSE) {

  if (weight.type == "normal") {
  # Normal distribution
  a <- seq(from = 1, to = n, by = 1)
  fmean <- seq(from = 1, to = n, by = 1)
  mat <- t(mapply(function(mean,sd) dnorm(a,mean,sd)/max(dnorm(a,mean,sd)), mean=fmean, sd=standard.deviation))
  if (weight.penalty == TRUE) {
    mat = -(1-mat)
    diag(mat) = 1
  }
  if (plot.weights == TRUE) {
    plot(mat[,1], type = "l", xaxt = "n", xlab = "Category", ylab = "Weight")
    axis(1, at = 1:length(mat[,1]))
    if (min(mat)<0) {abline(h=0, lty = "dotted")}
  }
  return(mat)
  }

  else if (weight.type == "arithmetic") {
  # Arithmetic progression
  mat = ((n-1)-abs(outer(seq(0, (n-1), 1), seq(0, (n-1), 1), `-`)))/(n-1)
  if (weight.penalty == TRUE) {
    mat = -(1-mat)
    diag(mat) = 1
  }
  if (plot.weights == TRUE) {
    plot(mat[,1], type = "l", xaxt = "n", xlab = "Category", ylab = "Weight")
    axis(1, at = 1:length(mat[,1]))
    if (min(mat)<0) {abline(h=0, lty = "dotted")}
  }
  return(mat)
  }

  else if (weight.type == "geometric") {
  # Geometric progression
  mult = geometric.multiplier
  mat = (abs(outer(seq(0, (n-1), 1), seq(0, (n-1), 1), `-`)))+1
  x=mult^seq(0,(n-1),by=1)
  x_n = (x-min(x))/(max(x)-min(x))
  if (mult > 1){
    x_dict = 1-x_n
  } else if (mult > 0 && mult < 1) {
    x_dict = x_n
  } else if (mult == 1) {
    x_dict = 1-seq(0, (n-1), 1)/(n-1)
  } else if (mult <= 0) {
    stop("Please enter a multiplier value greater than zero.")
  }
  for (i in 1:n) {
    mat[mat==i] = x_dict[i]
  }
  if (weight.penalty == TRUE) {
    mat = -(1-mat)
    diag(mat) = 1
  }
  if (plot.weights == TRUE) {
    plot(mat[,1], type = "l", xaxt = "n", xlab = "Category", ylab = "Weight")
    axis(1, at = 1:length(mat[,1]))
    if (min(mat)<0) {abline(h=0, lty = "dotted")}
  }
  return(mat)
  }

  else if (weight.type == "interval") {
  # Interval weight
  hi = interval.high
  lo = interval.low
  mat = (abs(outer(seq(0, (n-1), 1), seq(0, (n-1), 1), `-`)))+1
  x=seq(hi, lo, length.out = n)
  for (i in 1:n) {
    mat[mat==i] = x[i]
  }
  if (plot.weights == TRUE) {
    plot(mat[,1], type = "l", xaxt = "n", xlab = "Category", ylab = "Weight")
    axis(1, at = 1:length(mat[,1]))
    if (min(mat)<0) {abline(h=0, lty = "dotted")}
  }
  return(mat)
  }

  else if (weight.type == "custom") {
  # Custom weights
  wt = custom.weights
  mat = (abs(outer(seq(0, (n-1), 1), seq(0, (n-1), 1), `-`)))+1
  for (i in 1:n) {
    mat[mat==i] = wt[i]
  }
  if (plot.weights == TRUE) {
    plot(mat[,1], type = "l", xaxt = "n", xlab = "Category", ylab = "Weight")
    axis(1, at = 1:length(mat[,1]))
    if (min(mat)<0) {abline(h=0, lty = "dotted")}
  }
  return(mat)
  }

}
