% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cache.R
\name{cache_exec}
\alias{cache_exec}
\title{Cache the execution of an expression in memory or on disk}
\usage{
cache_exec(expr, path = "cache/", id = NULL, ...)
}
\arguments{
\item{expr}{An R expression to be cached.}

\item{path}{The path to save the cache. The special value \code{":memory:"} means
in-memory caching. If it is intended to be a directory path, please make
sure to add a trailing slash.}

\item{id}{A stable and unique string identifier for the expression to be used
to identify a unique copy of cache for the current expression from all
cache files (or in-memory elements). If not provided, an MD5 digest of the
\link{deparse}d expression will be used, which means if the expression does not
change (changes in comments or white spaces do not matter), the \code{id} will
remain the same. This may not be a good default is two identical
expressions are cached under the same \code{path}, because they could overwrite
each other's cache when one expression's cache is invalidated, which may or
may not be what you want. If you do not want that to happen, you need to
manually provide an \code{id}.}

\item{...}{More arguments to control the behavior of caching (see
\sQuote{Details}).}
}
\value{
If the cache is found, the cached value of the expression will be
loaded and returned (other local variables will also be lazy-loaded into
the current environment as a side-effect). If cache does not exist, the
expression is executed and its value is returned.
}
\description{
Caching is based on the assumption that if the input does not change, the
output will not change. After an expression is executed for the first time,
its result will be saved (either in memory or on disk). The next run will be
skipped and the previously saved result will be loaded directly if all
external inputs of the expression remain the same, otherwise the cache will
be invalidated and the expression will be re-executed.
}
\details{
Arguments supported in \code{...} include:
\itemize{
\item \code{vars}: Names of local variables (which are created inside the expression).
By default, local variables are automatically detected from the expression
via \code{\link[codetools:codetools]{codetools::findLocalsList()}}. Locally created variables are cached along
with the value of the expression.
\item \code{hash} and \code{extra}: R objects to be used to determine if cache should be
loaded or invalidated. If (the MD5 hash of) the objects is not changed, the
cache is loaded, otherwise the cache is invalidated and rebuilt. By default,
\code{hash} is a list of values of global variables in the expression (i.e.,
variables created outside the expression). Global variables are automatically
detected by \code{\link[codetools:findGlobals]{codetools::findGlobals()}}. You can provide a vector of names to
override the automatic detection if you want some specific global variables
to affect caching, or the automatic detection is not reliable. You can also
provide additional information via the \code{extra} argument. For example, if the
expression reads an external file \code{foo.csv}, and you want the cache to be
invalidated after the file is modified, you may use \code{extra = file.mtime("foo.csv")}.
\item \code{keep}: By default, only one copy of the cache corresponding to an \code{id}
under \code{path} is kept, and all other copies for this \code{id} is automatically
purged. If \code{TRUE}, all copies of the cache are kept. If \code{FALSE}, all copies
are removed, which means the cache is \emph{always} invalidated, and can be useful
to force re-executing the expression.
\item \code{rw}: A list of functions to read/write the cache files. The list is of the
form \code{list(load = function(file) {}, save = function(x, file) {})}. By
default, \code{\link[=readRDS]{readRDS()}} and \code{\link[=saveRDS]{saveRDS()}} are used. This argument can also take a
character string to use some built-in read/write methods. Currently available
methods include \code{rds} (the default), \code{raw} (using \code{\link[=serialize]{serialize()}} and
\code{\link[=unserialize]{unserialize()}}), and \code{qs} (using \code{\link[qs:qread]{qs::qread()}} and \code{\link[qs:qsave]{qs::qsave()}}). The \code{rds}
and \code{raw} methods only use base R functions (the \code{rds} method generates
smaller files because it uses compression, but is often slower than the \code{raw}
method, which does not use compression). The \code{qs} method requires the
\pkg{qs} package, which can be much faster than base R methods and also
supports compression.
}
}
\examples{
# the first run takes about 1 second
y1 = xfun::cache_exec({
    x = rnorm(1e+05)
    Sys.sleep(1)
    x
}, path = ":memory:", id = "sim-norm")

# the second run takes almost no time
y2 = xfun::cache_exec({
    # comments won't affect caching
    x = rnorm(1e+05)
    Sys.sleep(1)
    x
}, path = ":memory:", id = "sim-norm")

# y1, y2, and x should be identical
stopifnot(identical(y1, y2), identical(y1, x))
}
