% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CDVineCopulaConditional.r
\name{CDVineCondSim}
\alias{CDVineCondSim}
\title{Simulation from a conditional C- or D-vine}
\usage{
CDVineCondSim(RVM, Condition, N)
}
\arguments{
\item{RVM}{An \code{\link[VineCopula]{RVineMatrix}} object containing the information of the d-dimensional C- or D-Vine 
model (for further details about \code{\link[VineCopula]{RVineMatrix}} objects see the documentation file of the \code{VineCopula} package). 
If the full copula is 2-dimensional, RVM can be an \code{\link[VineCopula]{RVineMatrix}} object or a data.frame (or list) object 
where \code{$family}, \code{$par} and \code{$par2} are specified.}

\item{Condition}{A \code{N x Nx} matrix of the Nx conditioning variables.
For D-vine: data corresponding to the conditioning variable whose index is in \code{RVM$Matrix[i,i]}, are in i-th column of \code{Condition}. 
For C-vine: data corresponding to the conditioning variable whose index is in \code{RVM$Matrix[i,i]}, are in [(d+1)-i]-th column 
of \code{Condition}. See examples.}

\item{N}{Number of data to be simulated. By default N is taken from \code{Condition}, which is a \code{N x Nx} matrix.
It is necessary to specify \code{N} only when \code{Condition} is not given.}
}
\value{
A \code{N x d} matrix of the simulated variables from the given C- or D-vine copula model. In the first columns there are
the simulated conditioned variables, and in the last columns the conditioning variables \code{Condition}.
For more details about the exact order of the variables in the columns see the examples. The 
function is built to work easily in combination with \code{\link{CDVineCondFit}}.
}
\description{
Simulates from a d-dimensional conditional C- or D-vine of the variables (\strong{Y},\strong{X}),  
given the fixed conditioning variables \strong{X}. The algorithm works 
for vines satysfying the requirements discussed in \emph{Bevacqua et al. (2017)}. The algorthm implemented here 
is a modified version of those form \emph{Aas et al. (2009)} and is shown in \emph{Bevacqua et al. (2017)}.
}
\examples{

# Example 1: conditional sampling from a C-Vine

# Read data
data(dataset) 
data <- dataset$data[1:400,1:4]

# Define the variables Y and X. X are the conditioning variables, 
# which have to be positioned in the last columns of the data.frame
colnames(data) <- c("Y1","Y2","X3","X4")

\dontrun{
# Select a vine and fit the copula families, specifying that there are 2 conditioning variables
RVM <- CDVineCondFit(data,Nx=2,type="CVine")

# Set the values of the conditioning variables as those used for the calibration. 
# Order them with respect to RVM$Matrix, considering that is a C-Vine
d=dim(RVM$Matrix)[1]
cond1 <- data[,RVM$Matrix[(d+1)-1,(d+1)-1]]
cond2 <- data[,RVM$Matrix[(d+1)-2,(d+1)-2]]
condition <- cbind(cond1,cond2)

# Simulate the variables
Sim <- CDVineCondSim(RVM,condition)

# Plot the simulated variables over the observed
Sim <- data.frame(Sim)
overplot(Sim,data)



# Example 2: conditional sampling from a D-Vine

# Read data
data(dataset) 
data <- dataset$data[1:100,1:4]

# Define the variables Y and X. X are the conditioning variables, 
# which have to be positioned in the last columns of the data.frame
colnames(data) <- c("Y1","Y2","X3","X4")

# Select a vine and fit the copula families, specifying that there are 2 conditioning variables
RVM <- CDVineCondFit(data,Nx=2,type="DVine")
summary(RVM) #It is a D-Vine.

# Set the values of the conditioning variables as those used for the calibration. 
# Order them with respect to RVM$Matrix, considering that is a D-Vine.
cond1 <- data[,RVM$Matrix[1,1]]
cond2 <- data[,RVM$Matrix[2,2]]
condition <- cbind(cond1,cond2)

# Simulate the variables
Sim <- CDVineCondSim(RVM,condition)

# Plot the simulated variables over the observed
Sim <- data.frame(Sim)
overplot(Sim,data)



# Example 3

# Read data
data(dataset) 
data <- dataset$data[1:100,1:2]
colnames(data) <- c("Y1","X2")

# Fit copula
require(VineCopula)
BiCop <- BiCopSelect(data$Y1,data$X2)
BiCop

# Fix conditioning variable to low values and simulate
condition <- data$X2/10
Sim <- CDVineCondSim(BiCop,condition)

# Plot the simulated variables over the observed
Sim <- data.frame(Sim)
overplot(Sim,data)
}

}
\author{
Emanuele Bevacqua
}
\references{
Bevacqua, E., Maraun, D., Hobaek Haff, I., Widmann, M., and Vrac, M.: Multivariate statistical modelling of compound events via pair-copula constructions: analysis of floods in Ravenna (Italy), 
Hydrol. Earth Syst. Sci., 21, 2701-2723, https://doi.org/10.5194/hess-21-2701-2017, 2017.
\href{https://www.researchgate.net/publication/317414374_Multivariate_statistical_modelling_of_compound_events_via_pair-copula_constructions_Analysis_of_floods_in_Ravenna_Italy}{[link]} 
\href{https://www.hydrol-earth-syst-sci.net/21/2701/2017/hess-21-2701-2017.html}{[link]} 

Aas, K., Czado, C., Frigessi, A. and Bakken, H.: Pair-copula constructions of multiple dependence, Insurance:
Mathematics and Economics, 44(2), 182-198, <doi:10.1016/j.insmatheco.2007.02.001>, 2009. 
\href{http://www.sciencedirect.com/science/article/pii/S0167668707000194}{[link]} 

Ulf Schepsmeier, Jakob Stoeber, Eike Christian Brechmann, Benedikt Graeler, Thomas 
Nagler and Tobias Erhardt (2017). VineCopula: Statistical Inference of Vine Copulas. R 
package version 2.1.1. \href{https://CRAN.R-project.org/package=VineCopula}{[link]}
}
\seealso{
\code{\link{CDVineCondFit}}
}

