% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CJIVE_Functions.R
\name{cc.jive}
\alias{cc.jive}
\title{Canonical (Correlation) JIVE}
\usage{
cc.jive(
  dat.blocks,
  signal.ranks = NULL,
  joint.rank = 1,
  perc.var = 0.95,
  perm.test = TRUE,
  center = FALSE,
  nperms = 1000
)
}
\arguments{
\item{dat.blocks}{a list of two matrices with samples along rows and features along columns, which contain data on the same n individuals/sampling units}

\item{signal.ranks}{a vector of length two which contains the rank for the signal within each data block. The rank corresponds to the number of principal
components (PCs) to be retained within each data block. If NULL, the ranks are determined by the parameter 'perc.var.' Default is NULL}

\item{joint.rank}{The rank of the joint subspace i.e., number of components in the joint subspace}

\item{perc.var}{an alternative to signal.ranks that allows specification of ranks based on the desired proportion of total variation to be retained. F
For perc.var = p (where 0<p<1), rank is determined as the minimum number of eigenvalues whose cumulative sum is at least p*(total sum of eigenvalues)
Default is 0.95 (i.e. 95\% of total variation preserved for each data block).}

\item{perm.test}{logical (TRUE/FALSE) of whether permutation test for joint rank should be performed. Overrides 'joint.rank' parameter if TRUE. Default is TRUE}

\item{center}{logical (TRUE/FALSE) indicating whether data should be column-centered prior to testing. Default is TRUE}

\item{nperms}{integer value indicating the number of permutations that should be performed. Default is 1000}
}
\value{
A list of two lists:
        1) 'CanCorRes' contains results from the canonical correlation of PC scores including, the joint rank, joint subject sores,
        canonical correlations (and their respective p-values if perm.test was used), canonical loadings for the joint subspace, and total signal ranks
        2) 'sJIVE', i.e. Simple JIVE results, correspond to the AJIVE when all ranks are known; includes the joint and individual signal matrices, concatenated PC scores,
        and the projection matrix used to project each data block onto the joint subspace
}
\description{
Performs Canonical JIVE as described in the CJVE manuscript. This method is equivalent to AJIVE for 2 data sets.
}
\examples{
#Assign sample size and the number of features in each dataset
n = 200 #sample size
p1 = 100 #Number of features in data set X1
p2 = 100 #Number of features in data set X2

# Assign values of joint and individual signal ranks
r.J = 1 #joint rank
r.I1 = 2 #individual rank for data set X1
r.I2 = 2 #individual rank for data set X2


# Simulate data sets
ToyDat = GenerateToyData(n = 200, p1 = p1, p2 = p2, JntVarEx1 = 0.05, JntVarEx2 = 0.05,
                          IndVarEx1 = 0.25, IndVarEx2 = 0.25, jnt_rank = r.J, equal.eig = FALSE,
                          ind_rank1 = r.I1, ind_rank2 = r.I2, SVD.plots = TRUE, Error = TRUE,
                          print.cor = TRUE)
# Store simulated data sets in an object called 'blocks'
blocks <- ToyDat$'Data Blocks'

# Save Subject scores as R objects
JntScores = ToyDat[['Scores']][['Joint']]
IndivScore.X = ToyDat[['Scores']][["Indiv_1"]]
IndivScore.Y = ToyDat[['Scores']][["Indiv_2"]]

# Save joint variable loadings as R objects
JntLd.X = t(ToyDat$Loadings$Joint_1)
JntLd.Y = t(ToyDat$Loadings$Joint_2)

# Save individual variable loadings as R objects
IndivLd.X =t(ToyDat$Loadings$Indiv_1)
IndivLd.Y = t(ToyDat$Loadings$Indiv_2)

# Save joint, individual, and noise signal matrices as R objects
JX = ToyDat[[1]]$J1
JY = ToyDat[[1]]$J2
IX = ToyDat[[1]]$I1
IY = ToyDat[[1]]$I2
EX = ToyDat[[1]]$E1
EY = ToyDat[[1]]$E2


## Check that proportions of variation explained are (approximately) equal to intended values
JVE.X = MatVar(JX)/MatVar(blocks[[1]])
JVE.Y = MatVar(JY)/MatVar(blocks[[2]])

IVE.X = MatVar(IX)/MatVar(blocks[[1]])
IVE.Y = MatVar(IY)/MatVar(blocks[[2]])

TotVE.X = MatVar((JX + IX))/MatVar(blocks[[1]])
TotVE.Y = MatVar((JY + IY))/MatVar(blocks[[2]])


CJIVE.res = cc.jive(blocks, c(r.I1,r.I2)+r.J, r.J, perm.test = FALSE)
# CJIVE signal matrix estimates
J.hat = CJIVE.res$sJIVE$joint_matrices
I.hat = CJIVE.res$sJIVE$indiv_matrices

# CJIVE loading estimates
WJ = lapply(J.hat, function(x) x[['v']])
WI = lapply(I.hat, function(x) x[['v']])

# Plots of CJIVE estimates against true counterparts and include an estimate of their chordal norm
layout(matrix(1:6,2, byrow = TRUE))
plot(JntScores, CJIVE.res$CanCorRes$Jnt_Scores, xlab = "True Joint Scores",
    ylab = "CJIVE Joint Scores",
    sub = paste0("Chordal Norm = ",
                 round(chord.norm.diff(JntScores, CJIVE.res$CanCorRes$Jnt_Scores), 3)))
plot(JntLd.X, WJ[[1]][,1], xlab = "True Joint Loadings X", ylab = "CJIVE Joint Loadings X",
    sub = paste0("Chordal Norm = ", round(chord.norm.diff(JntLd.X, WJ[[1]][,1]), 3)))
plot(JntLd.Y, WJ[[2]][,1], xlab = "True Joint Loadings Y", ylab = "CJIVE Joint Loadings Y",
    sub = paste0("Chordal Norm = ", round(chord.norm.diff(JntLd.Y, WJ[[2]][,1]), 3)))
plot.new(); legend("left", paste("Comp.", 1:2), pch = 1, col  = c("orange", "green"),bty = "n" )
plot(IndivLd.X, WI[[1]][,1:2], xlab = "True Individual Loadings X",
    ylab = "CJIVE Individual Loadings X",
    col = c(rep("orange",p1), rep("green",p2)),
    sub = paste0("Chordal Norm = ", round(chord.norm.diff(IndivLd.X, WI[[1]][,1:2]), 3)))
plot(IndivLd.Y, WI[[2]][,1:2], xlab = "True Individual Loadings Y",
    ylab = "CJIVE Individual Loadings Y",
    col = c(rep("orange",p1), rep("green",p2)),
    sub = paste0("Chordal Norm = ", round(chord.norm.diff(IndivLd.Y, WI[[2]][,1:2]), 3)))
layout(1)

}
