% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hello.R
\name{cluster_map}
\alias{cluster_map}
\title{Create Cluster Map Based on Local Moran's I}
\usage{
cluster_map(
  dataset,
  lisa_value,
  lisa_label,
  label,
  lisa_cutoff,
  location_var,
  location_name,
  level2 = NULL,
  id_start = 0,
  comparison = ">",
  min_area = 0,
  min_ = 5,
  title = "Clusters Based on Queen Contiguity",
  subtitle = "",
  footnote = "",
  legend_position = "bottom",
  color_scheme = "C"
)
}
\arguments{
\item{dataset}{A spatial dataset of class `sf`.}

\item{lisa_value}{The name of the variable in the dataset containing Local Moran's I values.}

\item{lisa_label}{The name of the variable in the dataset containing the LISA label.}

\item{label}{The specific label to filter clusters.}

\item{lisa_cutoff}{A numeric value specifying the cutoff for LISA values.}

\item{location_var}{The variable name indicating the primary location in the dataset.}

\item{location_name}{The name of the variable for the location names.}

\item{level2}{An optional second level of location hierarchy. Default is `NULL`.}

\item{id_start}{The starting value for cluster IDs. Default is `0`.}

\item{comparison}{The comparison operator for filtering (`>`, `<`, `>=`, etc.). Default is `>`.}

\item{min_area}{Minimum area required for a cluster to be considered valid. Default is `0`.}

\item{min_}{Minimum number of districts required for a cluster to be valid. Default is `5`.}

\item{title}{The title of the map. Default is `"Clusters Based on Queen Contiguity"`.}

\item{subtitle}{A subtitle for the map. Default is `""`.}

\item{footnote}{A footnote for the map. Default is `""`.}

\item{legend_position}{The position of the legend on the map. Default is `"bottom"`.}

\item{color_scheme}{The color scheme for the map. Default is `"C"`.}
}
\value{
A list with the following components:
\item{dataset_with_clusters}{An `sf` object containing the dataset with assigned cluster IDs.}
\item{summary_clusters}{A data frame summarizing cluster information, including regions, area, and the number of locations.}
\item{plot}{A `ggplot` object for visualizing the clusters.}
}
\description{
This function creates a map of clusters based on Local Moran's I values. It identifies
clusters using Queen contiguity and visualizes them on a map.
}
\examples{
if (requireNamespace("spData", quietly = TRUE)) {
  library(sf)
  library(dplyr)
  library(spdep)
  library(spData)
  library(ggplot2)

  # Load US states data from spData
  us_states <- spData::us_states

  # Simplify for demonstration: Select a subset of columns
  us_states_data <- us_states \%>\%
    select(GEOID, NAME) \%>\%
    mutate(mean_wealth = rnorm(nrow(us_states), 50, 10))  # Add mock data

  # Define a shapefile path
  shapefile_path <- tempfile(fileext = ".shp")
  sf::st_write(us_states, shapefile_path, quiet = TRUE)

  # Corrected listw function call using your package
  us_states_listw <- DHSr::listw(
    shapefile_path = shapefile_path,
    data = us_states_data \%>\% sf::st_drop_geometry(),  # Drop geometry for compatibility
    loc_shape = "GEOID",
    loc_data = "GEOID",
    weight_function = function(d) exp(-d / 0.2)
  )

  # Apply Spdeplisa function
  lisa_result <- DHSr::Spdeplisa(
    data = us_states_data,
    variable_name = "mean_wealth",
    listw = us_states_listw
  )

  # Add LISA labels
  lisa_result <- lisa_result \%>\%
    mutate(lisa_label = case_when(
      lisa_I > 0 ~ "High-High",
      lisa_I < 0 ~ "Low-Low",
      TRUE ~ "Others"
    ))

  # Apply cluster_map function
  cluster_map_result <- DHSr::cluster_map(
    dataset = lisa_result,
    lisa_value = "lisa_I",
    lisa_label = "lisa_label",
    label = "High-High",
    lisa_cutoff = 0.5,
    location_var = "GEOID",
    location_name = "NAME",
    id_start = 1,
    comparison = ">",
    min_area = 0,
    min_ = 3,  # Reduced for smaller demonstration
    title = "Clusters Based on Queen Contiguity",
    subtitle = "High-High Clusters",
    footnote = "Generated using DHSr package",
    legend_position = "bottom",
    color_scheme = "C"
  )

  # View the resulting dataset with clusters
  head(cluster_map_result$dataset_with_clusters)

  # View the cluster summary
  print(cluster_map_result$summary_clusters)

  # Plot the clusters
  print(cluster_map_result$plot)
}
}
