#' Two-Sample CQ Test for High-Dimensional Covariance Matrices
#'
#' Performs a two-sample test to compare the covariance matrices of two high-dimensional samples.
#' This test is designed for situations where the number of variables \( p \) is large relative to the sample sizes \( n_1 \) and \( n_2 \).
#'
#' The test statistic is based on the difference between the sample covariance matrices,
#' normalized by their variances. The p-value can be computed using a suitable distribution.
#'
#' @param X A numeric matrix representing the first sample, where rows are variables and columns are observations.
#' @param Y A numeric matrix representing the second sample, where rows are variables and columns are observations.
#'
#' @return A list containing:
#' \item{statistics}{The test statistic \( Q_n \).}
#' \item{p.value}{The p-value of the test.}
#' \item{alternative}{The alternative hypothesis ("two.sided").}
#' \item{method}{The method used ("Two-Sample CQ test").}
#'
#' @examples
#' # Example usage:
#' set.seed(123)
#' p <- 50
#' n1 <- 20
#' n2 <- 20
#' X <- matrix(rnorm(n1 * p), nrow = p, ncol = n1)
#' Y <- matrix(rnorm(n2 * p), nrow = p, ncol = n2)
#' result <- CQ2(X, Y)
#' print(result)
#'
#' @export
CQ2 = function(X,Y) {
  p = nrow(X)
  n1 = ncol(X)
  n2 = ncol(Y)
  P1 = t(X) %*% X
  T1 = sum(P1)-sum(diag(P1))
  P2 = t(Y) %*% Y
  T2 = sum(P2)-sum(diag(P2))
  T3 = sum(sum(t(X) %*% Y))
  Tn = T1/(n1*(n1-1))+T2/(n2*(n2-1))-2*T3/(n1*n2)

  tres <- function(X) {
  p <- nrow(X)
  n <- ncol(X)
  return(sum((X %*% t(X))^2) - sum(diag((X %*% t(X))^2)))}
  s1 = tres(X)
  s2 = tres(Y)
  tres12 <- function(X, Y) {
  p <- nrow(X)
  n1 <- ncol(X)
  n2 <- ncol(Y)
  return(sum((t(X) %*% Y)^2))}
  s12 = tres12(X,Y)
  sn1 = 2*s1/(n1*(n1-1)) + 2*s2/(n2*(n2-1)) + 4*s12/(n1*n2)
  # Test statistic
  Qn = Tn/sqrt(sn1)
  names(Qn) = "Statistic"
  # P-value calculation (using normal approximation)
  p.value <- 2 * pnorm(-abs(Qn))
  res = list(statistics = Qn, p.value = p.value, alternative = "two.sided",method = "Two-Sample CQ test")
  return(res)
}

