% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{EpiTest.fit}
\alias{EpiTest.fit}
\title{Inference and Test}
\usage{
EpiTest.fit(Ancestry, Pheno, Trait, ParentName, Parents = NULL, Weight = FALSE)
}
\arguments{
\item{Ancestry}{a numeric matrix with 0/1 entries representing the allele ancestries
of a bi-parental population at a set of markers (0: homozygous for parent A alleles and
1: homozygous for parent B alleles)}

\item{Pheno}{a data.frame with a \code{Genotype} character column indicating the names
of individuals, a numeric column for each trait to be analyzed, and possibly
a \code{Weight} numeric column indicating the number of observation for each
individual. Missing values are discarded if present for a trait}

\item{Trait}{a string indicating the name of the phenotypic trait}

\item{ParentName}{a character vector of length 2 providing the parent names where the
first element should be named P0 (parent whose allele ancestries are coded 0) and
the second element should be named P1 (parent whose allele ancestries are coded 1)}

\item{Parents}{(optional) a data.frame with a \code{Genotype} character column indicating
the names of parents, a \code{Family} character column indicating the names of
families/populations, a numeric column for each trait to be analyzed, and possibly
a \code{Weight} numeric column indicating the number of observation for each parent
(see Details)}

\item{Weight}{a boolean indicating whether weights indicating the number of
observations should be used for the inference. If TRUE, a column \code{Weight}
should be found in Pheno and Parents (see Details)}
}
\value{
A list of four items: Beta, Sigma2, Test_fixed, Test_random
}
\description{
This function fits the EpiTest model on a bi-parental population
dataset, then outputs the estimates of the fixed and random effects, along
with their associated test results.
}
\details{
Depending on cases, the data may be organized in two different ways.

In Case 1, the \code{Ancestry} matrix and the \code{Pheno} data.frame respectively provide
the whole ancestry and phenotypic data, including the parental data.

In Case 2, the phenotypic data are split into two parts, the parental data being included
in a separate \code{Parents} data.frame.

In both cases, one may provide a set of weights associated to each individual by including
a \code{Weight} column in the \code{Pheno} and \code{Parents} data.frames. These weights
correspond to the number of observations that where used to compute the parental BLUPS/mean values.
These weights must be provided as a \code{Weight} column in the \code{Parents} and \code{Pheno}
data.frame.

The fitted EpiTest model includes three variance components: the segregation variance,
the (segregation \code{x} segregation) variance and the error variance.
For each genetic variance a likelihood ratio test is performed.

The fitted EpiTest model includes three fixed parameters: the intercept,
the linear regression coefficient (beta) on parent proportions and the quadratic
regression coefficient (delta) on parent proportions  that only involves epistatic effects.
Each fixed parameter is tested using a Wald test.

Additionally, epistasis can be tested by testing that both the (segregation \code{x} segregation) variance component
and the quadratic mean component (delta) are null through a likelihood ratio test. Note that in this
case the LRT is based on the full (i.e. unrestricted) likelihoods.

The function outputs a list of five items: Beta is the vector of estimated
fixed effects, Sigma2 is the vector of estimated variances, Test_full, Test_fixed,
Text_random are 3 data.table that provide the results of the tests for no epistasis (jointly tested on fixed and variance
components), and for the fixed and
random effects, respectively.
}
\examples{
## One bi-parental population, no weighting and no parental phenotypes
data(Pheno.list)
data(Ancestry.list)
Ancestry <- Ancestry.list[[1]]
Pheno <- Pheno.list[[1]]
ParentName <- c(P0 = 'B73',P1 = 'B97')
ETest.1 <- EpiTest.fit(Ancestry = Ancestry,
                       Pheno = Pheno,
                       ParentName = ParentName,
                       Trait = "GDD_DTA")

## One bi-parental population, with weights and parental phenotypes
data(Parents)
ETest.2 <- EpiTest.fit(Ancestry = Ancestry,
                       Pheno = Pheno,
                       ParentName = ParentName,
                       Trait = "GDD_DTA",
                       Parents = Parents,
                       Weight=TRUE)

## Full NAM analysis, with weights and parental phenotypes
Parent.list <- Parents$Genotype[-1]
names(Parent.list) <- Parents$Family[-1]
ETest.nam <- purrr::imap(Parent.list, ~ EpiTest.fit(Ancestry = Ancestry.list[[.y]],
                                                    Pheno = Pheno.list[[.y]],
                                                    ParentName=c(P0 = 'B73',P1 = .x),
                                                    Parents = Parents,
                                                    Trait = 'GDD_DTA',
                                                    Weight = TRUE))
}
