% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SMART.R
\name{SMART}
\alias{SMART}
\title{\code{SMART} - Scoring Metric after Permutation}
\description{
\code{SMART} estimates the importance of a feature to the clustering algorithm
by measuring changes in cluster assignments by scoring functions after
permuting selected feature. Cluster-specific \code{SMART} indicates the importance
of specific clusters versus the remaining ones, measured by a binary scoring
metric. Global \code{SMART} assigns importance scores across all clusters, measured
by a multi-class scoring metric. Currently, \code{SMART} can only be used for hard
label predictors.
}
\details{
Let \eqn{M \in \mathbb{N}_0^{k \times k}} denote the multi-cluster
confusion matrix and \eqn{M_c \in \mathbb{N}_0^{2 \times 2}} the binary
confusion matrix for cluster c versus the remaining clusters. \code{SMART} for
feature set S corresponds to:
\deqn{
\text{Multi-cluster scoring:} \quad \text{SMART}(X, \tilde{X}_S) = h_{\text{multi}}(M) \\
\text{Binary scoring:} \quad \text{SMART}(X, \tilde{X}_S) = \text{AVE}(h_{\text{binary}}(M_1), \dots, h_{\text{binary}}(M_k))
}
where \eqn{\text{AVE}} averages a vector of binary scores, e.g., via micro or
macro averaging.
In order to reduce variance in the estimate from shuffling the data, one can
shuffle t times and evaluate the distribution of scores. Let \eqn{\tilde{X}_S^{(t)}}
denote the t-th shuffling iteration for feature set S. The \code{SMART} point
estimate is given by: \cr
\deqn{
\overline{\text{SMART}}(X, \tilde{X}_S) = \psi\left(\text{SMART}(X, \tilde{X}_S^{(1)}),
 \dots, \text{SMART}(X, \tilde{X}_S^{(t)})\right)
}
where \eqn{\psi} extracts a sample statistic such as the mean or median or quantile.
}
\examples{
\donttest{
# load data and packages
require(factoextra)
require(FuzzyDBScan)
multishapes = as.data.frame(multishapes[, 1:2])
# Set up an train FuzzyDBScan
eps = c(0, 0.2)
pts = c(3, 15)
res = FuzzyDBScan$new(multishapes, eps, pts)
res$plot("x", "y")
# create hard label predictor
predict_part = function(model, newdata) model$predict(new_data = newdata, cmatrix = FALSE)$cluster
predictor = ClustPredictor$new(res, as.data.frame(multishapes), y = res$clusters,
                               predict.function = predict_part, type = "partition")
# Run SMART globally
macro_f1 = SMART$new(predictor, n.repetitions = 50, metric = "f1", avg = "macro")
macro_f1 # print global SMART
macro_f1$plot(log = TRUE) # plot global SMART
# Run cluster specific SMART
classwise_f1 = SMART$new(predictor, n.repetitions = 50, metric = "f1")
macro_f1 # print regional SMART
macro_f1$plot(log = TRUE) # plot regional SMART
}
}
\seealso{
\link[iml:FeatureImp]{iml::FeatureImp}

\link{SMART}

\link{SMART}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{avg}}{(\code{character(1)} or \code{NULL})\cr
\code{NULL} is calculating cluster-specific (binary)
metrics. \code{"micro"} summarizes binary scores to a global
score that treats each instance in the data set with equal
importance. \code{"macro"} summarizes binary scores to a global
score that treats each cluster with equal importance.}

\item{\code{metric}}{\code{character(1)}\cr
The binary similarity metric used.}

\item{\code{predictor}}{\link{ClustPredictor}\cr
The object (created with \code{ClustPredictor$new()}) holding
the cluster algorithm and the data.}

\item{\code{data.sample}}{\link{data.frame}\cr
The data, including features and cluster soft/ hard labels.}

\item{\code{sampler}}{any\cr
Sampler from the \code{predictor} object.}

\item{\code{features}}{(\verb{character or list})\cr
Features/ feature sets to calculate importance scores.}

\item{\code{n.repetitions}}{(\code{numeric(1)})\cr
How often is the shuffling of the feature repeated?}

\item{\code{results}}{(\code{data.table})\cr
A \link{data.table} containing the results from \code{SMART} procedure.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-SMART-new}{\code{SMART$new()}}
\item \href{#method-SMART-print}{\code{SMART$print()}}
\item \href{#method-SMART-plot}{\code{SMART$plot()}}
\item \href{#method-SMART-clone}{\code{SMART$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-SMART-new"></a>}}
\if{latex}{\out{\hypertarget{method-SMART-new}{}}}
\subsection{Method \code{new()}}{
Create a \link{SMART} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{SMART$new(
  predictor,
  features = NULL,
  metric = "f1",
  avg = NULL,
  n.repetitions = 5
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{predictor}}{\link{ClustPredictor}\cr
The object (created with \code{ClustPredictor$new()}) holding
the cluster algorithm and the data.}

\item{\code{features}}{(\verb{character or list})\cr
For which features do you want importance scores calculated. The default
value of \code{NULL} implies all features. Use a named list of character vectors
to define groups of features for which joint importance will be calculated.}

\item{\code{metric}}{\code{character(1)}\cr
The binary similarity metric used. Defaults to \code{f1},
where F1 Score is used. Other possible binary scores are
\code{"precision"}, \code{"recall"}, \code{"jaccard"}, \code{"folkes_mallows"}
and \code{"accuracy"}.}

\item{\code{avg}}{(\code{character(1)} or \code{NULL})\cr
Either \code{NULL}, \code{"micro"} or \code{"macro"}.
Defaults to \code{NULL} is calculating cluster-specific (binary)
metrics. \code{"micro"} summarizes binary scores to a global
score that treats each instance in the data set with equal
importance. \code{"macro"} summarizes binary scores to a global
score that treats each cluster with equal importance.
For unbalanced clusters, \code{"macro"} is more recommendable.}

\item{\code{n.repetitions}}{(\code{numeric(1)})\cr
How often should the shuffling of the feature be repeated?
The higher the number of repetitions the more stable and
accurate the results become.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
(data.frame)\cr
data.frame with the results of the feature importance computation.
One row per feature with the following columns:
For global scores:
\itemize{
\item importance.05 (5\% quantile of importance values from the repetitions)
\item importance (median importance)
\item importance.95 (95\% quantile) and the permutation.error (median error
over all repetitions).
For cluster specific scores each column indicates for a different cluster.
}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-SMART-print"></a>}}
\if{latex}{\out{\hypertarget{method-SMART-print}{}}}
\subsection{Method \code{print()}}{
Print a \code{SMART} object
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{SMART$print()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
\code{character} \cr
Information about \code{predictor}, \code{data}, \code{metric}, and \code{avg}
and head of the \code{results}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-SMART-plot"></a>}}
\if{latex}{\out{\hypertarget{method-SMART-plot}{}}}
\subsection{Method \code{plot()}}{
plots the similarity score results of a \code{SMART}
object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{SMART$plot(log = FALSE, single_cl = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{log}}{\code{logical(1)} \cr
Indicator weather results should be logged. This can be
useful to distinguish the importance if similarity scores
are all close to 1.}

\item{\code{single_cl}}{\code{character(1)} \cr
Only used for cluster-specific scores (\code{avg = NULL}).
Should match one of the cluster names.
In this case, importance scores for a single cluster are
plotted.}
}
\if{html}{\out{</div>}}
}
\subsection{Details}{
The plot shows the similarity per feature.
For global scores:
When \code{n.repetitions} in \code{SMART$new} was larger than 1, then we get
multiple similarity estimates per feature. The similarity are aggregated and
the plot shows the median similarity per feature (as dots) and also the
90\%-quantile, which helps to understand how much variance the computation has
per feature.
For cluster-specific scores:
Stacks the similarity estimates of all clusters per feature.
Can be used to achieve a global estimate as a sum of
cluster-wise similarities.
}

\subsection{Returns}{
ggplot2 plot object
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-SMART-clone"></a>}}
\if{latex}{\out{\hypertarget{method-SMART-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{SMART$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
