% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HMM_simulation.R
\name{HMM_simulation}
\alias{HMM_simulation}
\title{Generating Realizations of a Hidden Markov Model}
\usage{
HMM_simulation(
  size,
  m,
  delta = rep(1/m, times = m),
  gamma = 0.8 * diag(m) + rep(0.2/m, times = m),
  distribution_class,
  distribution_theta,
  obs_range = c(NA, NA),
  obs_round = FALSE,
  obs_non_neg = FALSE,
  plotting = 0
)
}
\arguments{
\item{size}{length of the time-series of hidden states and observations (also \code{T}).}

\item{m}{a (finite) number of states in the hidden Markov chain.}

\item{delta}{a vector object containing values for the marginal probability 
distribution of the \code{m} states of the Markov chain at the time point \code{t=1}.
Default is \code{delta = rep(1 / m, times = m)}.}

\item{gamma}{a matrix (\code{ncol = nrow = m}) containing values for the transition 
matrix of the hidden Markov chain. 
Default is \code{gamma=0.8 * diag(m)} \code{ + rep(0.2 / m, times = m)}}

\item{distribution_class}{a single character string object with the abbreviated name of 
the \code{m} observation distributions of the Markov dependent observation process.  
The following distributions are supported by this algorithm: Poisson (\code{pois}); 
generalized Poisson (\code{genpois}); normal (\code{norm}); geometric (\code{geom}).}

\item{distribution_theta}{a list object containing the parameter values for the 
\code{m} observation distributions that are dependent on the hidden Markov state.}

\item{obs_range}{a vector object specifying the range for the observations to be 
generated.  For instance, the vector \code{c(0,1500)} allows only observations 
between 0 and 1500 to be generated by the HMM.  Default value is \code{FALSE}.  
See Notes for further details.}

\item{obs_round}{a logical object. \code{TRUE} if all generated observations are natural.
Default value is \code{FALSE}. See Notes for further details.}

\item{obs_non_neg}{a logical object. \code{TRUE}, if non negative observations are 
generated. Default value is \code{FALSE}.  See Notes for further details.}

\item{plotting}{a numeric value between 0 and 5 (generates different outputs). 
NA suppresses graphical output. Default value is \code{0}.\cr
  \code{0}: output 1-5 \cr
  \code{1}: summary of all results \cr
  \code{2}: generated time series of states of the hidden Markov chain  \cr
  \code{3}: means (of  the observation distributions, which depend on the states of the 
            Markov chain) along the time series of states of the hidden Markov chain \cr
  \code{4}: observations along the time series of states of the hidden Markov chain \cr
  \code{5}: simulated observations}
}
\value{
The function \code{ HMM_simulation } returns a list containing the following components:
\describe{
The function \code{ HMM_simulation } returns a list containing the following components:
\item{size}{length of the generated time-series of hidden states and observations.}
\item{m}{input number of states in the hidden Markov chain.}
\item{delta}{a vector object containing the chosen values for the marginal probability 
     distribution of the \code{m} states of the Markov chain at the time point \code{t=1}.}
\item{gamma}{a matrix containing the chosen values for the transition matrix of the 
     hidden Markov chain.}
\item{distribution_class}{a single character string object with the abbreviated name of 
     the chosen observation distributions of the Markov dependent observation process.}
\item{distribution_theta}{a list object containing the chosen values for the parameters 
     of the \code{m} observation distributions that are dependent on the hidden Markov state.}
\item{markov_chain}{a vector object containing the generated sequence of states of the 
     hidden Markov chain of the HMM.}
\item{means_along_markov_chain}{a vector object containing the sequence of means 
     (of the state dependent distributions) corresponding to the generated sequence of states.}
\item{observations}{a vector object containing the generated sequence of (state dependent) 
     observations of the HMM.}
}
}
\description{
This function generates a sequence of hidden states of a Markov chain and a 
corresponding parallel sequence of observations.
}
\note{
Some notes regarding the default values: \cr
\code{gamma}: \cr
The default setting assigns higher probabilities for remaining in a state than c
hanging into another.  
\code{obs_range}:\cr
Has to be used with caution. since it manipulates the results of the HMM. 
If a value for an observation at time \code{t} is generated outside the defined range, 
it will be regenerated as long as it falls into \code{obs_range}. It is possible just 
to define one boundary, e.g. \code{obs_range=c(NA,2000)} for observations lower than 
2000, or \code{obs_range=c(100,NA)} for observations higher than 100. 
\code{obs_round }:\cr
Has to be used with caution!  Rounds each generated observation and hence manipulates 
the results of the HMM (important for the normal distribution based HMM). 
\code{obs_ non_neg}:\cr
Has to be used with caution, since it manipulates the results of the HMM. If a negative 
value for an observation at a time \code{t} is generated, it will be re-generated as 
long as it is non-negative (important for the normal distribution based HMM).
}
\examples{
# i.) Generating a HMM with Poisson-distributed data -----
\donttest{
 Pois_HMM_data <- 
   HMM_simulation(size = 300, 
                     m = 5, 
    distribution_class = "pois", 
    distribution_theta = list( lambda=c(10,15,25,35,55)))
    
 print(Pois_HMM_data)
 }
 
 # ii.) Generating 6 physical activities with normally -----
 #      distributed accelerometer counts using a HMM.
 
 # Define number of time points (1440 counts equal 6 hours of 
 # activity counts assuming an epoch length of 15 seconds).
 size <- 1440
 
 # Define 6 possible physical activity ranges
 m <- 6
 
 # Start with the lowest possible state 
 # (in this case with the lowest physical activity)
 delta <- c(1, rep(0, times = (m - 1)))
 
 # Define transition matrix to generate according to a 
 # specific activity 
 gamma <- 0.935 * diag(m) + rep(0.065 / m, times = m)
 
 # Define parameters 
 # (here: means and standard deviations for m=6 normal 
 #  distributions that define the distribution in 
 #  a phsycial acitivity level)
 distribution_theta <- list(mean = c(0,100,400,600,900,1200), 
                            sd = rep(x = 200, times = 6))
                            
 # Assume for each count an upper boundary of 2000
 obs_range <-c(NA,2000)
 
 # Accelerometer counts shall not be negative
 obs_non_neg <-TRUE
 
 # Start simulation
 \donttest{
 accelerometer_data <- 
     HMM_simulation(size = size, 
                       m = m, 
                   delta = delta, 
                   gamma = gamma, 
      distribution_class = "norm", 
      distribution_theta = distribution_theta, 
               obs_range = obs_range, 
             obs_non_neg = obs_non_neg, 
                plotting = 0)
 print(accelerometer_data)
 }
}
\seealso{
\code{\link{AIC_HMM}}, \code{\link{BIC_HMM}}, \code{\link{HMM_training}}
}
\author{
Vitali Witowski (2013).
}
\keyword{datagen}
