#' @title Generate Synthetic data
#'
#' @description
#' Synthetic data generated using copulas and marginal distributions,
#' with the purpose of exampling the functions of the package. Three options are given: "noTrend",
#' "marginalTrend", and "dependenceStructure".
#'
#' The generated "noTrend" data follows a Clayton copula with fixed Kendall Tau (\eqn{\tau=0.2}).
#' "marginalTrend", follows the same copula, however the variables follow a Generalized Extreme Value
#' distribution with fixed scale and shape parameters (\eqn{\sigma=1, \xi=-0.1}),
#' the location is linearly non-stationary with \eqn{\mu_X = 0.05\cdot t} and \eqn{\mu_Y = 0.07\cdot t}.
#' Finally "dependenceStructure" presents trend in the dependence structure, the data
#' was generated from a Clayton copula with a linear non-stationary \eqn{\tau} parameter.
#' For more information we refer the reader to the source material by Goutali and Chebana (2024).
#'
#' @param trend Character vector, the trend of the generated data. Options are "noTrend",
#'  "marginalTrend", and "dependenceStructure".
#' @param n Integer, the number of data points to be generated. Must be between 10
#'  and 200, default is 100.
#'
#' @return A dataset of dimensions \eqn{n \times 2} with the generated data.
#'
#' @references
#' Goutali, D., and Chebana, F. (2024). Multivariate overall and dependence trend tests, applied to hydrology,
#' Environmental Modelling & Software, 179, \doi{10.1016/j.envsoft.2024.106090}
#'
#' @examples
#' # NO TREND
#' generate_data("noTrend", n = 50)
#'
#' # TREND IN BOTH MARGINALS
#' generate_data("marginalTrend", n = 50)
#'
#' # TREND IN DEPENDENCE STRUCTURE
#' generate_data("dependenceStructure", n = 50)
#'
#' @importFrom copula rCopula claytonCopula iTau
#' @importFrom VGAM qgev
#'
#' @export

generate_data <- function(trend = c("noTrend", "marginalTrend", "dependenceStructure"), n = 100) {}

generate_data <- function(trend = c("noTrend", "marginalTrend", "dependenceStructure"), n = 100) {

  trend <- match.arg(trend)

  if (!is.numeric(n) || length(n) != 1) {
    stop("Parameter 'n' must be numeric")
  }

  if ((n %% 1) != 0 || n < 10 || n > 200) {
    stop("Parameter 'n' must be an integer between 10 and 200")
  }

  switch(trend,
         "noTrend" = {
           coefs <- rep(0.2, n)
           data <- generator(coefs, n)
         },
         "marginalTrend" = {
           coefs <- rep(0.2, n)
           data <- generator(coefs, n, marginal = TRUE)
         },
         "dependenceStructure" = {
           coefs <- seq(0, 1, length.out = n)
           data <- generator(coefs, n)
         })

  return(data)
}

generator <- function(coefs, n, marginal = FALSE) {

  dependence <- sapply(coefs, function(tau) {
    claytonCopula(iTau(claytonCopula(), tau), dim = 2)})

  if (length(unique(coefs)) == 1) {
    if (marginal) {

      margin1 <- numeric(n)
      margin2 <- numeric(n)

      for (j in 1:n) {
        u <- rCopula(1, dependence[[j]])
        margin1[j] <- qgev(u[1], location = 0.05 * j, scale = 1, shape = -0.1)
        margin2[j] <- qgev(u[2], location = 0.07 * j, scale = 1, shape = -0.1)
      }
      data <- cbind(margin1, margin2)
    } else {
      data <- matrix(NA, nrow = n, ncol = 2)
      for (j in 1:n) {
        data[j, ] <- rCopula(1, dependence[[j]])
      }
    }

  } else {
    data <- t(sapply(1:n, function(j) rCopula(1, dependence[[j]])))
  }

  colnames(data) <- c("X", "Y")
  return(data)
}
