\name{pcplus}
\alias{pcplus}

\title{Piecewise constant plus smooth estimation}
\description{Computes the \acronym{PCpluS} estimator \cite{F. Pein (2021)} for a given bandwidth and the penalty lambda. Bandwidth and lambda can be obtained by cross-validation using the function \code{\link{cv.pcplus}}. The \acronym{PCpluS} estimator returns a piecewise constant function plus a smooth function as well as the change-points of the piecewise constant function.}

\usage{
pcplus(y, bandwidth, lambda, sd = NULL, nlambda = 30L, thresh = 1e-7, maxit = 1e5L)
}

\arguments{
\item{y}{a numeric vector containing the observations, only finite values are allowed}
\item{bandwidth}{a single positive value specifying the bandwidth for the kernel smoother; must be between \code{1 / length(n)} and \code{0.25} or \code{Inf}, smaller values are replaced by \code{1 / n} and larger by \code{Inf} with a warning; see \cite{F. Pein (2021)} for an interpretation of \code{bandwidth == Inf}}
\item{lambda}{a single positive numeric or a decreasing sequence of positive numeric values giving the penalty for the fused lasso. If a sequence is passed, then only the smallest value is used to compute the estimator. However, passing a sequence of lambda values is often much faster than passing a single value}
\item{sd}{a single positive value giving the standard deviation of the observations; may be \code{NULL}, in which case a robust estimator is used}
\item{nlambda}{a single positive integer specifying the number of lambda values to pass to the lasso; only used if \code{lambda} is a single value, in which case an exponentially decreasing sequence is generated, with \code{lambda} being the smallest value. As explained for \code{lambda}, only this value is used for the estimator, but adding other values may reduce the run time}
\item{thresh}{a single positive numeric value giving a convergence threshold for coordinate descent. Each inner coordinate-descent loop continues until the maximum change in the objective after any coefficient update is less than \code{thresh} times the null deviance}
\item{maxit}{a single positive integer giving the maximum number of passes over the data for all lambda values}
}

\value{
a \code{\link{list}} containing the entries \code{est} with the fitted values of the esitmator, \code{smooth} with the smooth part of the estimator, \code{cpfun} with the change-point part of the estimator, and \code{cps} with the estimated change-point locations.
}

\seealso{\code{\link{cv.pcplus}}}

\references{
Pein, F. (2021). Change-point regression with a smooth additive disturbance. \emph{arXiv preprint} arXiv:2112.03878.
} 

\examples{
library(PCpluS)
set.seed(1)
y <- c(rnorm(125), rnorm(125, 3)) + sin(2 * pi * 1:250 / 250)

CV <- cv.pcplus(y)
ret <- pcplus(y, lambda = CV$lambda, bandwidth = CV$bandwidth)

plot(y, pch = 16)
lines(ret$est, col = "red")
abline(v = ret$cps)
}

\keyword{nonparametric}
