% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/misc.R
\name{k1fun}
\alias{k1fun}
\title{Constant K1}
\usage{
k1fun(dCdtheta, fun, data, empirical, mean = 0, sd = 1)
}
\arguments{
\item{dCdtheta}{a function providing the limit as \eqn{\theta \to a} of the
copula derivative with respect to \eqn{\theta}. For the readily available copulas, namely, \code{frank}, \code{amh},
\code{fgm} and  \code{gauss}, \eqn{a=0}.}

\item{fun}{optionally, a function providing an estimator for the probability density function.}

\item{data}{the observed time series. Only used to obtain the quantile function
when \code{empirical = TRUE}.}

\item{empirical}{logical. If \code{TRUE}, the sample estimators for the density
and quantile functions are considered. Otherwise, the gaussian density and
quantile functions are used instead.}

\item{mean}{the mean of the gaussian distribution.
Only used if \code{empirical = FALSE}}

\item{sd}{the standard deviation of the gaussian distribution.
Only used if \code{empirical = FALSE}}
}
\value{
The value of \eqn{K_1}.
}
\description{
Calculates an estimate for the constant \eqn{K_1} given by
\deqn{K_1 = \int_0^1\int_0^1\frac{1}{l_0(u)l_n(v)}\lim_{\theta\to a}\frac{\partial C_{\theta}(u,v)}{\partial\theta}\,dudv, }
where \eqn{l_m(x):= F_m'\big(F_m^{(-1)}(x)\big)}, \eqn{a} is such that \eqn{C_a(u,v)=uv} (the product copula), and
\eqn{\{F_n\}_{n \geq 0}} is a sequence of absolutely continuous distribution
functions.
}
\details{
Here \eqn{F'} and \eqn{F^{(-1)}} are replaced by sample estimators for these
functions or the gaussian density and quantile functions are used, depending
on the context.

The function \code{\link{kdens}} is used as sample estimator of \eqn{F'} and
\code{\link[stats]{quantile}} is the sample estimator of \eqn{F^{(-1)}}.
}
\examples{
trunc = 50000
cks <- arfima.coefs(d = 0.25, trunc = trunc)
eps <- rnorm(trunc+1000)
x <- sapply(1:1000, function(t) sum(cks*rev(eps[t:(t+trunc)])))

# kernel density function
dfun <- kdens(x)

# calculating K1 using four copulas and empirical estimates for F' and F^{(-1)}
K1_frank_e <- k1fun(dCdtheta = dCtheta_frank, fun = dfun,
                 data = x, empirical = TRUE)
K1_amh_e <- k1fun(dCdtheta = dCtheta_amh, fun = dfun,
                 data = x, empirical = TRUE)
K1_fgm_e <- k1fun(dCdtheta = dCtheta_fgm, fun = dfun,
                 data = x, empirical = TRUE)
K1_gauss_e <- k1fun(dCdtheta = dCtheta_gauss, fun = dfun,
                 data = x, empirical = TRUE)

# calculating K1 using four copulas and gaussian marginals
K1_frank_g <- k1fun(dCdtheta = dCtheta_frank, fun = NULL, data = NULL,
                  empirical = FALSE, mean = mean(x), sd = sd(x))
K1_amh_g <- k1fun(dCdtheta = dCtheta_amh, fun = NULL, data = NULL,
                  empirical = FALSE, mean = mean(x), sd = sd(x))
K1_fgm_g <- k1fun(dCdtheta = dCtheta_fgm, fun = NULL, data = NULL,
                  empirical = FALSE, mean = mean(x), sd = sd(x))
K1_gauss_g <- k1fun(dCdtheta = dCtheta_gauss, fun = NULL, data = NULL,
                  empirical = FALSE, mean = mean(x), sd = sd(x))

# comparing results
 data.frame(MARGINAL = c("Empirical", "Gaussian"),
            FRANK = c(K1_frank_e, K1_frank_g),
            AMH = c(K1_amh_e,  K1_amh_g),
            FGM = c(K1_fgm_e, K1_fgm_g),
            GAUSS = c(K1_gauss_e, K1_gauss_g))

}
