% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FunctionsPoDParameters.R
\name{PoDParamEstimation}
\alias{PoDParamEstimation}
\title{PoD curve parameters estimation}
\usage{
PoDParamEstimation(diseasedTiters,
                   nondiseasedTiters,
                   nondiseasedGenerationCount,
                   repeatCount = 500,
                   adjustTiters = FALSE,
                   adjustFrom = log2(10),
                   adjustTo = log2(5))
}
\arguments{
\item{diseasedTiters}{numeric vector: all diseased titers, subject level data}

\item{nondiseasedTiters}{numeric vector: non-diseased titers from immunogenicity subset, subject level data}

\item{nondiseasedGenerationCount}{numeric: total number of non-diseased subjects in the clinical trial}

\item{repeatCount}{numeric: how many times is the dataset bootstrapped and the PoD curve parameter estimation performed}

\item{adjustTiters}{boolean: set to TRUE if titer values should be adjusted, for details see \code{PoD} function}

\item{adjustFrom}{numeric: value specifying the detection limit, all values below the detection limit will be adjusted to adjustTo value}

\item{adjustTo}{numeric: value to which titers below the detection limit will be adjusted}
}
\value{
results: PoD curve parameters after resetting the disease status, named data.frame of estimated PoD curve parameters (pmax, slope, et50); see details for more information

resultsPriorReset: PoD curve parameters prior to resetting the status, named data.frame of estimated PoD curve parameters (pmax, slope, et50); see details for more information

failcount: number of iterations in which MLE failed to converge; see details for more information
}
\description{
Function estimates the PoD curve parameters (pmax, slope, et50) using \code{PoDMLE} function. Number of PoD curves estimated equals to the repeatCount input parameter.

The estimation is performed using provided diseased and non-diseased subject level data.
}
\details{
diseasedTiters: subject level titers of all diseased in the clinical trial

nondiseasedTiters: subject level titers of non-diseased subjects in the immunogenicity subset

There are two possible scenarios

\itemize{
  \item Full: Full information about non-diseased titers is available, i.e subject level data for all non-diseased subjects from the clinical trial (nondiseasedGenerationCount = number of all non-diseased subjects in the clinical trial).

  \item Ratio or Fixed: Information about non-diseased titers is available only for the immunogenicity subset. In order to compensate for these missing titers we upsampling of this subset to the total number of non-diseased (nondiseasedGenerationCount) in the trial is needed. 

}

nondiseasedGenerationCount: number of all non-diseased subjects in the clinical trial

NOTE: Number of estimated parameters can be lower than repeatCount as MLE does not necessary converge in all estimations; failcount (number of iterations in which MLE failed to converge) is also returned; for details see \code{MLE} function.


Function steps

\itemize{
  \item Upsample non-diseased if needed (needed for methods Ratio and Fixed) - from immunogenicity subset size (N = NondiseasedImmunogenicitySubset$N) to the whole trial size (N = nondiseasedGenerationCount). For details see \code{GenerateNondiseased} function.

  \item Estimate PoD curve: resultsPriorReset

  \item Reset disease status: the purpose is to estimate the confidence intervals of the PoD curve and its parameters

  Part of the reset disease status procedure is the non-parametric bootstrap: titers of diseased and non-diseased subjects are pooled, and associated PoDs are calculated using their titer values and estimated PoD curve. Based on the subject level probabilities (PoDs), the disease status is reestimated.

  \item Re-estimate PoD curve: new diseased and non-diseased titers are used to reestimate the PoD curve
}
}
\examples{
## Data preparation
data(diseased)
data(nondiseased)

## Example 1
# Creating imunogenicity subset, method = "Full"
NondiseasedImmunogenicitySubset <- 
    ImmunogenicitySubset(diseased, 
                         nondiseased, 
                         method = list(name = "Full", 
                                       value = "NA"))

# Number of all non-diseased subjects in the clinical trial
nondiseasedGenerationCount <- nondiseased$N

PoDParamEstimation(diseased$titers,
                   NondiseasedImmunogenicitySubset$titers,
                   nondiseasedGenerationCount,
                   repeatCount = 10)

## Example 2
# Creating imunogenicity subset, method = "Ratio", value = 4
NondiseasedImmunogenicitySubset <- 
    ImmunogenicitySubset(diseased, 
                         nondiseased, 
                         method = list(name = "Ratio", 
                                       value = 4))
                                       
# Number of all non-diseased subjects in the clinical trial
nondiseasedGenerationCount <- nondiseased$N

PoDParamEstimation(diseased$titers,
                   NondiseasedImmunogenicitySubset$titers,
                   nondiseasedGenerationCount,
                   repeatCount = 10)

}
