% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ProbitSpatial.R
\name{ProbitSpatialFit}
\alias{ProbitSpatialFit}
\title{Fit a spatial probit model.}
\usage{
ProbitSpatialFit(formula,data,W,
         DGP='SAR',method="conditional",varcov="varcov",
         M=NULL,control=list())
}
\arguments{
\item{formula}{an object of class \code{formula}: a symbolic 
description of the model to be fitted.}

\item{data}{the data set containing the variables of the model.}

\item{W}{the spatial weight matrix of class \code{"dgCMatrix"}.}

\item{DGP}{the data generating process of \code{data}: SAR, SEM, SARAR 
(Default is SAR).}

\item{method}{the optimisation method: \code{"conditional"} or 
\code{"full-lik"} (Defaul is \code{"conditional"}, see Details).}

\item{varcov}{the likelihood function is computed using the 
variance-covariance matrix (\code{"varcov"}) or the precision matrix 
(\code{"precision"})? Default is \code{"varcov"}.}

\item{M}{the second spatial weight matrix for SARAR models. Same class as W.}

\item{control}{a list of control parameters. See Details.}
}
\value{
Return an object of class \code{ProbitSpatial}.
}
\description{
Approximate likelihood estimation of the probit model with spatial 
autoregressive (SAR), spatial error (SEM), spatial autoregressive with 
autoregressive disturbances (SARAR).
}
\details{
The estimation is based on the approximate value of the true likelihood of 
	spatial probit models. 
	The DGP of the spatial autoregressive model (SAR) model is the following
	\deqn{y = \rho Wy + X\beta + \epsilon,}
	where the disturbances \eqn{\epsilon} are iid standard normally distributed, 
	\eqn{W} is a sparse spatial weight matrix and \eqn{\rho} is the spatial lag 
	parameter. The variance of the error term is equal 
	to \eqn{\Sigma=\sigma^2((I_n-\rho W)^{-1}((I_n-\rho W)^{-1})^{t})}.
	The DGP of the spatial error model (SEM) is as follows
	\deqn{y = X\beta+u,}
	\deqn{u = \rho W u + \epsilon,}
	where the disturbances \eqn{\epsilon} are iid standard normally distributed, 
	\eqn{W} is a sparse spatial weight matrix and \eqn{\rho} is the spatial 
error  parameter. The variance of the error term  
is equal to \eqn{\Sigma=\sigma^2((I_n-\rho W)^{-1}((I_n-\rho W 
)^{-1})^{t})}.
	The DGP of the spatial autoregressive model with autoregressive disturbances 
	(SARAR) is as follows
	\deqn{y = \rho Wy + X\beta + u,}
	\deqn{u = \lambda M u + \epsilon,}
	where the disturbances \eqn{\epsilon} are iid standard normally distributed, 
	\eqn{W} and \eqn{M} are two sparse spatial weight matrix, while \eqn{\rho} 
	 and \eqn{\lambda} are the spatial lag and spatial error parameters, 
	 respectively. The variance of the error term  
is equal to \eqn{\Sigma=\sigma^2((I_n-\rho W)^{-1}(I_n-\lambda 
M)^{-1}((I_n-\lambda M)^{-1})^{t}((I_n-\rho W)^{-1})^{t})}.

	The approximation is inspired by the Mendell-Elston approximation 
	of the multivariante normal probabilities (see References). It makes use of 
	the Cholesky decomposition of the variance-covariance matrix \eqn{\Sigma}.

	The \code{ProbitSpatialFit} command estimates the model by maximising the 
	approximate log-likelihood. We propose two optimisation method:
	\describe{
 \item{\code{"conditional"}:}{ it relies on a standard probit estimation  
	which applies to the model estimated 
	conditional on \eqn{\rho}.}
 \item{\code{"full-lik"}:}{ it minimises the full-log-likelihood using the 
	analytical gradient functions (only available for SAR and SEM 
	specification). The optimisation is performed by means of the 
	\code{\link[stats]{optim}} function with \code{method = "BFGS"}.}
	}
	In both cases a \code{"conditional"} estimation is performed. If 
	\code{method="conditional"}, then \code{ProbitSpatialFit} returns 
	the results of this first estimation. In case \code{method="full-lik"},
	the function tries to improve the log-likelihood by means of a further 
	exploration around the value of the parameters found by the conditional 
step.
	The conditional step is usually very accurate and particularly fast. The 
	second step is more time consuming and does not always improve the results
	of the first step. We dissuade the user from using the full-likelihood 
method 	for sample sizes bigger than ten thousands, since the computation of 
the gradients is quite slow.  Simulation studies reported in Martinetti and 
	Geniaux (2017) prove that the conditional estimation is highly reliable,
even  if compared to the full-likelihood ones.

	In order to reduce the computation time of the function
 \code{ProbitSpatialFit}, we propose a variant of the likelihood-function 
	estimation that uses the inverse of the variance-covariance matrix (a.k.a. 
	precision matrix). This variant applies to both the \code{"conditional"} and 
	the \code{"full-lik"} methods and can be invoked by setting 
	\code{varcov="precision"}. Simulation studies reported in Martinetti and 
	Geniaux (2017) suggest that the accuracy of the results with the precision 
	matrix are sometimes worst than the one with the true variance-covariance 
	matrix, but the estimation time is considerably reduced.

	The control argument is a list that can supply any of the following 
	components:
	\describe{
  \item{\code{iW_CL}}{the order of approximation of \eqn{(I_n-\rho W)^{-1}} 
	used in the \code{"conditional"} method. Default is 6, while 0 means no 
	approximation (it uses exact inversion of matrixes, not suitable for big 
	sample sizes). See Martinetti and Geniaux (2017) for further references.}
  \item{\code{iW_FL}}{the order of approximation of \eqn{(I_n-\rho W)^{-1}} 
	used in the computation of the likelihood function for the \code{"full-lik"} 
	method. Default is 0, meaning no approximation.}
  \item{\code{iW_FG}}{the order of approximation of \eqn{(I_n-\rho W)^{-1}} 
	used in the computation of the gradient functions for the \code{"full-lik"} 
	method. Default is 0, meaning no approximation.}
	\item{\code{reltol}}{relative convergence tolerance. It represents 	
\code{tol} in \code{\link[stats]{optimize}} function  for 
\code{method="conditional"} and  \code{reltol} in \code{\link[stats]{optim}} 
function for \code{method="full-lik"}. Default is 1e-5.}
  \item{\code{prune}}{the pruning value used in the gradients. Default is 0,
	meaning no pruning. Typacl values are around 1e-3 and 1e-6. They help 
	reducing the estimation time of the gradient functions.}
  \item{\code{silent}}{Default is TRUE.}
}
}
\examples{
\donttest{
n <- 1000
nneigh <- 3
rho <- 0.5
beta <- c(4,-2,1)
W <- generate_W(n,nneigh,seed=123)
X <- cbind(1,rnorm(n,2,2),rnorm(n,0,1))
colnames(X) <- c("intercept","X1","X2")
y <- sim_binomial_probit(W=W,X=X,beta=beta,rho=rho,model="SAR")
d <- as.data.frame(cbind(y,X))
mod <- ProbitSpatialFit(y~X1+X2,d,W,
       DGP='SAR',method="conditional",varcov="varcov")
}
}
\references{
\describe{
\item{Mendell and Elston (1974)}{N. Mendell and R. Elston. Multifactorial 
	qualitative traits: genetic analysis and prediction of recurrence risks. 
\emph{Biometrics} 30, 41--57, 1974.}

\item{Martinetti and Geniaux (2017)}{D. Martinetti and G. Geniaux. 
	Approximate likelihood estimation of spatial probit models. \emph{Regional 
	Science and Urban Economics} 64, 30-45, 2017.}}
}
