% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/peer.ces.R
\name{cespeer.sim}
\alias{cespeer.sim}
\title{Simulating Peer Effect Models with a CES Social Norm}
\usage{
cespeer.sim(
  formula,
  Glist,
  parms,
  rho,
  lambda,
  beta,
  epsilon,
  structural = FALSE,
  init,
  tol = 1e-10,
  maxit = 500,
  data
)
}
\arguments{
\item{formula}{A formula object (\link[stats]{formula}): a symbolic description of the model. \code{formula} should be specified as, for example, \code{~ x1 + x2},
where \code{x1} and \code{x2} are control variables, which can include contextual variables such as averages or quantiles among peers.}

\item{Glist}{The adjacency matrix. For networks consisting of multiple subnets (e.g., schools), \code{Glist} must be a list of subnets, with the \code{m}-th element being an \eqn{n_m \times n_m} adjacency matrix, where \eqn{n_m} is the number of nodes in the \code{m}-th subnet.}

\item{parms}{A vector defining the true values of \eqn{(\rho, \lambda', \beta')'}, where \eqn{\rho} is the substitution parameter of the CES function and
\eqn{\lambda} is either the peer effect parameter for the reduced-form specification or a 2-vector with the first component being conformity peer effects and the second component representing total peer effects.
The parameters \eqn{\rho}, \eqn{\lambda}, and \eqn{\beta} can also be specified separately using the arguments \code{rho}, \code{lambda}, and \code{beta} (see the Details section of \code{\link{cespeer}}).}

\item{rho}{The true value of the substitution parameter of the CES function.}

\item{lambda}{The true value of the peer effect parameter \eqn{\lambda}. It must include conformity and total peer effects for the structural model.}

\item{beta}{The true value of the vector \eqn{\beta}.}

\item{epsilon}{A vector of idiosyncratic error terms. If not specified, it will be simulated from a standard normal distribution (see the model specification in the Details section of \code{\link{cespeer}}).}

\item{structural}{A logical value indicating whether simulations should be performed using the structural model. The default is the reduced-form model (see the Details section of \code{\link{cespeer}}).}

\item{init}{An optional initial guess for the equilibrium.}

\item{tol}{The tolerance value used in the Fixed Point Iteration Method to compute the outcome \code{y}. The process stops if the \eqn{\ell_1}-distance
between two consecutive values of \code{y} is less than \code{tol}.}

\item{maxit}{The maximum number of iterations for the Fixed Point Iteration Method.}

\item{data}{An optional data frame, list, or environment containing the model variables. If a variable is not found in \code{data}, it is retrieved from \code{environment(formula)}, typically the environment from which \code{cespeer.sim} is called.}
}
\value{
A list containing:
\item{y}{The simulated variable.}
\item{epsilon}{The idiosyncratic error.}
\item{init}{The initial guess.}
\item{iteration}{The number of iterations before convergence.}
}
\description{
\code{cespeer.sim} simulates peer effect models with a Constant Elasticity of Substitution (CES) based social norm (Boucher et al., 2024).
}
\examples{
set.seed(123)
ngr  <- 50
nvec <- rep(30, ngr)
n    <- sum(nvec)
G    <- lapply(1:ngr, function(z){
  Gz <- matrix(rbinom(nvec[z]^2, 1, 0.3), nvec[z])
  diag(Gz) <- 0
  Gz/rowSums(Gz) # Row-normalized network
})
tau  <- seq(0, 1, 0.25)
X    <- cbind(rnorm(n), rpois(n, 2))
l    <- 0.55
rho  <- 3
b    <- c(4, -0.5, 1)

out  <- cespeer.sim(formula = ~ X, Glist = G, rho = rho, lambda = l, beta = b)
summary(out$y)
out$iteration
}
\references{
Boucher, V., Rendall, M., Ushchev, P., & Zenou, Y. (2024). Toward a general theory of peer effects. \emph{Econometrica, 92}(2), 543-565, \doi{10.3982/ECTA21048}.
}
\seealso{
\code{\link{cespeer}}, \code{\link{qpeer.sim}}
}
