# Name   : sensitivity.analysis
# Desc   : Generic call for sensitivity analysis methods
# Date   : 2012/18/06
# Update : 2023/02/28
# Author : Boelle, Obadia
###############################################################################


#' @title
#' Sensitivity analysis of basic reproduction ratio to begin/end dates
#' 
#' @description
#' Sensitivity analysis of reproduction ratio using supported estimation methods.
#' 
#' @details
#' This is a generic call function to use either sa.time or sa.GT. 
#' Argument must be chosen accordingly to `sa.type`. Please refer to [sa.time()] 
#' and [sa.GT()] for further details about arguments.
#' 
#' `begin` and `end` vectors must have the same length for the sensitivity 
#' analysis to run. They can be provided either as dates or numeric values, 
#' depending on the other parameters (see [check.incid()]. If some `begin` or `end` 
#' dates overlap, they are ignored and corresponding uncomputed data are set to `NA`.
#' Also, note that unreliable Rsquared values are achieved for very small time 
#' periods (begin ~ end). These values are not representative of the epidemic 
#' outbreak behaviour.
#' 
#' @param incid A vector of incident cases.
#' @param GT Generation time distribution from [generation.time()].
#' @param begin Vector of begin dates for the estimation of epidemic.
#' @param end Vector of end dates for estimation of the epidemic.
#' @param est.method Estimation method used for sensitivity analysis.
#' @param sa.type String argument to choose between `time` and `GT` sensitivity analysis.
#' @param res If specified, will extract most of data from a `R0.R`-class result already generated by [estimate.R()] and run sensitivity analysis on it.
#' @param GT.type Type of distribution for GT (see GT.R for details).
#' @param GT.mean.range Range of mean values used for all GT distributions throughout the simulation. Must be provided as a vector.
#' @param GT.sd.range Range of standard deviation values used for GT distributions. Must be provided as a vector.
#' @param t Dates vector to be passed to estimation function.
#' @param date.first.obs Optional date of first observation, if t not specified.
#' @param time.step Optional. If date of first observation is specified, number of day between each incidence observation.
#' @param ... Parameters passed to inner functions
#' 
#' @return
#' A sensitivity analysis object of class `R0.S` with components depending on sensitivity analysis `sa.type`.
#' 
#' @export
#' 
#' @example tests/sensitivity.analysis.R
#' 
#' @author Pierre-Yves Boelle, Thomas Obadia



# Function declaration

sensitivity.analysis <- function(
    incid, 
    GT             = NULL, 
    begin          = NULL, 
    end            = NULL,  
    est.method     = NULL, 
    sa.type, 
    res            = NULL, 
    GT.type        = NULL, 
    GT.mean.range  = NULL, 
    GT.sd.range    = NULL, 
    t              = NULL, 
    date.first.obs = NULL, 
    time.step      = 1, 
    ... 
)


# Code

{
  # Depending on argument "sa.type", a different sub-routine is called, with varying input arguments
  if (is.null(sa.type)) {
    stop("argument sa.type should be either \"time\" or \"GT\".")
  }
  else if ((sa.type == "time") && (!is.null(res))) {
    sa.object <- sa.time(res = res, begin = begin, end = end, ...)
  }
  
  else if ((sa.type == "time") && (is.null(res))) {
    if ((is.null(GT)) | (is.null(est.method))) {
      stop("Missing input argument (probably GT or est.method). Please check sa.time documentation for further details.")
    }
    else {
      sa.object <- sa.time(incid = incid, GT = GT, begin = begin, end = end, est.method = est.method, t = t, date.first.obs = date.first.obs, time.step = time.step, res = res, ...)
    }
    
  }
  else if (sa.type == "GT") {
    if ((is.null(GT.type)) | (is.null(GT.mean.range)) | (is.null(GT.sd.range)) | (is.null(est.method))) {
      stop("Missing input argument (probably GT.type, GT.mean.range, GT.sd.range or est.method). Please check sa.time documentation for further details.")
    }
    else {
      sa.object <- sa.GT(incid = incid, GT.type = GT.type, GT.mean.range = GT.mean.range, GT.sd.range = GT.sd.range, begin = begin, end = end, est.method = est.method, t = t, date.first.obs = date.first.obs, time.step = time.step, ...)
    }
  }
  else {
    stop(paste("sa.type =",sa.type,"is not a valid argument. Must be \"time\" or \"GT\"."))
  }
  
  return(sa.object)
}
