% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rpygeo_reticulate.R
\name{rpygeo_load}
\alias{rpygeo_load}
\title{Load output of ArcPy functions into R session}
\usage{
rpygeo_load(data)
}
\arguments{
\item{data}{\code{reticulate} object or filename of the ArcPy function output}
}
\value{
\code{raster} or \code{sf} object
}
\description{
This function loads the output of an ArcPy function into the R session. Raster files are loaded as \code{raster} objects and vector files as \code{sf} objects.
}
\details{
Currently files and datasets stored in file geodatabases are supported.

Supported file formats:
\itemize{
  \item Tagged Image File Format (.tif)
  \item Erdas Imagine Images (.img)
  \item Esri Arc/Info Binary Grid (.adf)
  \item Esri ASCII Raster (.asc)
  \item Esri Shapefiles (.shp)
  }

Supported datasets:
\itemize{
  \item Feature Class
  \item Raster Dataset
}

Esri has not released an API for raster datasets in file geodatabases. \code{rpygeo_load} converts a raster dataset to a temporary ASCII raster first and then loads it into the R session. Be aware that this can take a long time for large raster datasets.

This function can be used with the \code{\%>\%} operator from the \code{dplyr} package. The \code{\%>\%} operator forwards the \code{reticulate} object from the ArcPy function to \code{rpygeo_load} (s. Example 1). If used without the \code{\%>\%} operator an \code{reticulate} object can be specified for the \code{data} parameter (s. Example 2). It is also possible to use the filename of the ArcPy function output (s. Example 3). For Arc/Info Binary Grids the \code{data} parameter is just the name of the directory, which contains the \code{adf} files.
}
\examples{

\dontrun{
# Load packages
library(RPyGeo)
library(magrittr)
library(RQGIS)
library(spData)

# Get data
data(dem, package = "RQGIS")
data(nz, package = "spData")

# Write data to disk
writeRaster(dem, file.path(tempdir(), "dem.tif"), format = "GTiff")
st_write(nz, file.path(tempdir(), "nz.shp"))

# Load the ArcPy module and build environment
arcpy <- arcpy_build_env(overwrite = TRUE, workspace = tempdir())

# Create a slope raster and load it into the R session (Example 1)
slope <-
  arcpy$Slope_3d(in_raster = "dem.tif", out_raster = "slope.tif") \%>\%
  rpygeo_load()

# Create a aspect raster and load it into the R session (Example 2)
ras_aspect <- arcpy$sa$Aspect(in_raster = "dem.tif")
rpygeo_load(ras_aspect)

# Convert elevation raster to polygon shapefile and load it into R session (Example 3)
arcpy$RasterToPolygon_conversion("dem.tif", "elev.shp")
rpygeo_load("elev.shp")
}

}
\author{
Marc Becker
}
