#' Resolvable PBIB designs
#'
#' @param v Total number of treatments
#' @param p Positive integer (>=2)
#'
#' @return This function generates a PBIB design and its parameters, variance factors and efficiency factor.

#' @description This function generates a new series of resolvable Partially Balanced Incomplete Block Designs (PBIBDs) and its parameters (v, b1, b2, r, k1, k2), canonical efficiency factors, variance factor between associates and average variance factors.
#' @examples
#' library(ResPBIBD)
#' PBIBD3(12, 4)
#' @export

PBIBD3=function(v,p){
  if(p>=3 && v==2*p*(p-1)){
    v=2*p*(p-1)
    mat1 = matrix(1:(v/2), nrow = p-1, ncol = p, byrow = T)
    n_rows = nrow(mat1)
    ################
    o<-c()
    for(i in 1:n_rows){
      for(j in 2:n_rows){
        if(i!=j){
          o<-c(o,i,j)
        }
      }
    }
    mat<-matrix(o,ncol=2,byrow=T)
    for(i in 1:nrow(mat)){
      mat[i,]<-sort(mat[i,])
    }

    comb_rows<-unique(mat)

    #############

    all = c(t(comb_rows))
    final_combn = matrix(,nrow=0,ncol=0)
    for(i in length(all)){
      for(j in all){
        final_combn = c(final_combn,mat1[j,])
      }
    }
    a1=matrix(final_combn,nrow=nrow(comb_rows),byrow=T)
    ######################################################
    v2 = 1+v/2
    z = c()
    while (v2 <= v) {
      x = c(v2)
      z = c(z, x)
      v2 = v2 + 1
    }
    mat2=matrix(z, nrow = p-1, ncol = p, byrow = T)
    n_rows = nrow(mat2)
    ########################
    o<-c()
    for(i in 1:n_rows){
      for(j in 2:n_rows){
        if(i!=j){
          o<-c(o,i,j)
        }
      }
    }
    mat<-matrix(o,ncol=2,byrow=T)
    for(i in 1:nrow(mat)){
      mat[i,]<-sort(mat[i,])
    }

    comb_rows<-unique(mat)
    ###############
    all = c(t(comb_rows))
    final_combn = matrix(,nrow=0,ncol=0)
    for(i in length(all)){
      for(j in all){
        final_combn = c(final_combn,mat2[j,])
      }
    }
    a2=matrix(final_combn,nrow=nrow(comb_rows),byrow=T)
    #########################################################
    a3=cbind(mat1, mat2)
    ########################################################
    a=rbind(a1, a2, a3)
    ########################################################
    row.names(a) = c(1:nrow(a))
    ss = 1
    while (ss <= nrow(a)) {
      rownames(a)[ss] <- paste("Block", as.character(ss),
                               sep = "")
      ss = ss + 1
    }
    #######################################################
    design=rbind(a1, a2, a3)
    b=nrow(design)
    k=ncol(design)

    N_mat = matrix(0, v, b)
    for (i in 1:b) {
      for (j in 1:k) {
        N_mat[design[i, j], i] = N_mat[design[i, j], i] + 1
      }
    }

    r=rowSums(N_mat)[1]
    K = diag(colSums(N_mat), b, b)
    R = diag(rowSums(N_mat), v, v)
    kvec = colSums(N_mat)
    Kinv = diag(1/kvec, nrow = b, ncol = b)
    C_mat = R - N_mat %*% Kinv %*% t(N_mat)
    E = eigen(C_mat, only.values = T)
    E1 = unlist(E)
    E_positive = E1[E1 >= 1e-09]
    n = length(E_positive)
    C_Efficiency = n/(r * sum(c(1/E_positive)))
    p_matrix = matrix(, nrow = 0, ncol = v)
    i = 1
    j = 1
    while (i <= (choose(v, 2))) {
      j = i + 1
      while (j <= v) {
        p1 = matrix(0, nrow = 1, ncol = v)
        p1[i] = 1
        p1[j] = -1
        p_matrix = rbind(p_matrix, p1)
        j = j + 1
      }
      i = i + 1
    }
    p_invC_Pprme = (p_matrix) %*% MASS::ginv(C_mat) %*% t(p_matrix)
    var = diag(p_invC_Pprme)
    var1 = round(var, digits = 4)
    var2 = unique(var1)
    Average_var = mean(var)
    A1 = c("Number of treatments",
           "Number of blocks",
           "Number of replications",
           "Size of  blocks")
    A2 = c("v", "b", "r", "k")
    A3 = c(v, b, r, k)
    A = cbind(A1, A2, A3)
    ###########################################################
    message("\n", "Design parameters")
    prmatrix(A, rowlab = , collab = rep("", ncol(A)),
             quote = FALSE, na.print = "")

    message("\n","Resolvable PBIB design")
    prmatrix(a, rowlab = , collab = rep("", ncol(a)),
             quote = FALSE, na.print = "")

    message("\n", "Canonical efficiency factor")
    print(round(C_Efficiency,4), quote = F)

    message("\n", "Variance factor between associates" )
    B1 <- c("variance factor between first associates",
            "variance factor between second associates",
            "variance factor between third associates",
            "variance factor between fourth associates")
    B2 <- c(var2[1], var2[2], var2[3], var2[4])
    B <- cbind(B1, B2)
    prmatrix(B, rowlab = , collab = rep("", ncol(B)), quote = FALSE, na.print = "")

    message("\n", "Average variance factor" )
    print(round(Average_var,4), quote = F)
  }
  else {
    message("Please enter v=2*p*(p-1), where p>=2")
  }
}
##########################


