% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RoughSets-package.R
\docType{package}
\name{RoughSets-package}
\alias{RoughSets-package}
\alias{RoughSets}
\alias{Introduction-RoughSets}
\alias{Introduction-FuzzyRoughSets}
\title{Getting started with the RoughSets package}
\description{
This part contains global explanations about the implementation and use of the \code{RoughSets} package.
The package \code{RoughSets} attempts to provide a complete tool to model and analyze
information systems based on rough set theory (RST) and fuzzy rough set theory (FRST).
From fundamental point of view, this package allows to construct rough sets by defining lower and upper approximations.
Furthermore, recent methods for tackling common tasks in data mining, such as data preprocessing (e.g., discretization, feature selection, missing value completion,
and instance selection), rule induction, and prediction classes or decision values of new datasets
are available as well.
}
\details{
There are two main parts considered in this package which are RST and FRST.
RST was introduced by (Pawlak, 1982; Pawlak, 1991) which provides sophisticated mathematical tools to model
and analyze information systems that involve uncertainty and imprecision. By employing indiscernibility relation among objects, RST does not require
additional parameters to extract information.
The detailed explanation about fundamental concepts of RST can be read in Section \code{\link{Introduction-RoughSets}}. Secondly, FRST, an extension of
RST, was introduced by (Dubois and Prade, 1990) as a combination between
fuzzy sets proposed by (Zadeh, 1965) and RST. This concept allows to analyze continuous attributes without
performing discretization on data first. Basic concepts of FRST
can be seen in \code{\link{Introduction-FuzzyRoughSets}}.

Based on the above concepts, many methods have been proposed and applied for dealing with several different domains.
In order to solve the problems, methods employ the indiscernibility relation and lower and upper approximation concepts.
All methods that have been implemented in this package will be explained by grouping based on their domains. The following is
a list of domains considered in this package:
\itemize{
\item Basic concepts: This part, we can divide into four different tasks which are
      indiscernibility relation, lower and upper approximations, positive region and discernibility matrix.
      All of those tasks have been explained briefly in Section \code{\link{Introduction-RoughSets}} and

     \code{\link{Introduction-FuzzyRoughSets}}.
\item Discretization: It is used to convert real-valued attributes into nominal/symbolic ones in an information system.
      In RST point of view, this task attempts to maintain the discernibility between objects.
\item Feature selection: It is a process for finding a subset of features which have the same quality as the complete feature set.
     In other words, its purpose is to select the significant features and eliminate the dispensible ones.
     It is a useful and necessary process when we are facing datasets containing large numbers of features. From RST and FRST perspective,
     feature selection refers to searching superreducts and reducts. The detailed information about reducts can be read in
     \code{\link{Introduction-RoughSets}} and \code{\link{Introduction-FuzzyRoughSets}}.
\item Instance selection: This process is aimed to remove noisy, superfluous, or inconsistent instances from training datasets but retain consistent ones.
     In other words, good accuracy of classification is achieved by removing instances which do not give positive contributions.
\item Prediction/classification: This task is used to predict decision values of a new dataset (test data).
     We consider implementing some methods to perform this task, such as fuzzy-rough nearest neighbor approaches, etc.
\item Rule induction: This task refers to generate IF - THEN rules. The rule represents knowledge which is contained in a dataset.
     One advantage of building rules is that naturally the model is easy to interpret. Then, predicted values over new datasets can be determined by
     considering the rules.
}

As we mentioned before, we have embedded many well-known algorithms or techniques for handling the above domains. The algorithms were considered
since experimentally it has been proven that they were able to tackle complex tasks. They are implemented as functions that were organized
to work with the same data structures. So, users can perform various approaches for a particular task easily and then compare their results.
In order to be recognized quickly, generally we have chosen the names of the functions with some conventions. The names contain three parts
which are \code{prefix}, \code{suffix}, and \code{middle} that are separated by a point. The following is a description of each
part.
\itemize{
\item \code{prefix}: There are some different prefixes for names of functions expressing a kind of task to be performed.
                     The function names with prefix \code{BC} refer to \emph{basic concepts} which means that the functions are created for
                     implementing the basic concepts of RST and FRST.
                     While prefix \code{D} refers to \emph{discretization}, \code{FS}, \code{IS}, \code{RI}, \code{MV}, and \code{C} refer to \emph{feature selection},
                     \emph{instance selection}, \emph{rule induction}, \emph{missing value completion}, and \emph{classifier based on nearest neighbor} domains. Furthermore, \code{SF} and \code{X} mean that
                     functions are used as \emph{supporting functions} which are not related directly with RST and FRST and \emph{auxiliary} functions which are called as a parameter.
\item \code{suffix}: It is located at the end of names. There are two types available: \code{RST} and \code{FRST}. \code{RST} represents \emph{rough set theory}
                     while \code{FRST} shows that the function is applied to \emph{fuzzy rough set theory}. Additionally, some functions that do not have
                     \code{RST} or \code{FRST} suffix are used for both theories.
\item \code{middle}: All other words in the middle of the names are used to express the actual name of a particular method/algorithm or functionality.
                     In this case, it could consist of more than one word separated by points.
}
For instance, the function \code{\link{BC.IND.relation.RST}} is used to calculate the indiscernibility relation which is one of the basic concepts of RST.
Other functions that have names not based on the above rules are S3 functions e.g., \code{summary} and \code{predict} which are
used to summarize objects and predict new data, respectively.

The following description explains domains and their algorithms implemented in the package:
\enumerate{
\item \bold{The implementations of RST}: This part outlines some considered algorihtms/methods based on RST.
             The approaches can be classified based on their tasks as follows:
            \enumerate{
            \item The basic concepts: The following is a list showing tasks and their implementations as functions.
                      \itemize{
                      \item Indiscernibility relation: It is a relation determining whether two objects are indiscernible by some attributes.
                            It is implemented in \code{\link{BC.IND.relation.RST}}.
                      \item Lower and upper approximations: These approximations show whether objects can be classified with certainty or not.
                            It is implemented in \code{\link{BC.LU.approximation.RST}}.
                      \item Positive region: It is used to determine objects that are included in positive region and the degree of dependency.
                            It is implemented in \code{\link{BC.positive.reg.RST}}.
                      \item Discernibility matrix: It is used to create a discernibility matrix showing attributes that discern each pair of objects.
                            It is implemented in \code{\link{BC.discernibility.mat.RST}}.
                      }
            \item Discretization: There are a few methods included in the package:
                     \itemize{
                      \item \code{\link{D.global.discernibility.heuristic.RST}}: It implements the global discernibility algorithm
                             which is computing globally semi-optimal cuts using the maximum discernibility heuristic.
					           \item \code{\link{D.discretize.quantiles.RST}}: It is a function used for computing cuts of the "quantile-based" discretization into \eqn{n} intervals.
                      \item \code{\link{D.discretize.equal.intervals.RST}}: It is a function used for computing cuts of the "equal interval size" discretization into \eqn{n} intervals.
                      }
     		The output of these functions is a list of cut values which are the values for converting real to nominal values.
             So, in order to generate a new decision table according to the cut values, we need to call \code{\link{SF.applyDecTable}}.
             Additionally, we have implemented \code{\link{D.discretization.RST}} as a wrapper function collecting all methods considered to perform discretization tasks.
            \item Feature selection: According to its output, it can be classified into the following groups:
                     \itemize{
                      \item Feature subset: It refers to a superreduct which is not necessarily minimal. In other words, the methods in this group
                             might generate just a subset of attributes.
                            \itemize{
                                 \item QuickReduct algorithm: It has been implemented in \code{\link{FS.quickreduct.RST}}.
                                 \item Superreduct generation: It is based on some criteria:
                                                   entropy, gini index, discernibility measure, size of positive region.

                                                   It is implemented in \code{\link{FS.greedy.heuristic.superreduct.RST}}.
                            }
                            Furthermore, we provide a wrapper function \code{\link{FS.feature.subset.computation}} in order to give a user interface for many methods of RST and FRST that are included in this group.
                      \item Reduct: The following are methods that produce a single decision reduct:
                            \itemize{
                                 \item Reduct generation based on criteria: It is based on different criteria which are
                                                   entropy, gini index, discernibility measure, size of positive region.
                                                   It has been implemented in \code{\link{FS.greedy.heuristic.reduct.RST}}.
                                 \item Permutation reduct: It is based on a permutation schema over all attributes.
                                                    It has been implemented in \code{\link{FS.permutation.heuristic.reduct.RST}}.
                             }
                            Furthermore, we provide a wrapper function \code{\link{FS.reduct.computation}} in order to give a user interface toward many methods of RST and FRST that are included in this group.
                      \item All reducts: In order to generate all reducts, we execute \code{\link{FS.all.reducts.computation}}. However,
                            before doing that, we need to call \code{\link{BC.discernibility.mat.RST}} for
                            constructing a decision-relative discernibility matrix
                      }
                      It should be noted that the outputs of the functions are decision reducts. So, for generating a new decision table according to the decision reduct,
                      we need to call \code{\link{SF.applyDecTable}}.
           \item Rule induction: We provide several functions used to generate rules, as follows:
                      \itemize{
                          \item The function \code{\link{RI.indiscernibilityBasedRules.RST}}: This function requires the output of the feature selection functions.
                          \item The function \code{\link{RI.CN2Rules.RST}}: It is a rule induction method based on the CN2 algorithm.
                          \item The function \code{\link{RI.LEM2Rules.RST}}: It implements a rule induction method based on the LEM2 algorithm.
                          \item The function \code{\link{RI.AQRules.RST}}: It is a rule induction based on the AQ-style algorithm.
                      }
                       After obtaining the rules, we execute \code{\link{predict.RuleSetRST}} considering our rules and given newdata/testing data to obtain predicted values/classes.
             }
\item \bold{The implementations of FRST}: As in the \code{RST} part, this part contains several algorithms that can be classified into several groups based on their purpose.
          The following is a description of all methods that have been implemented in functions:
 \enumerate{
  \item Basic concepts: The following is a list showing tasks and their implementations:
   \itemize{
      \item Indiscernibility relations: they are fuzzy relations determining to which degree two objects are similar.
            This package provides several types of relations which are implemented in a single function
            called \code{\link{BC.IND.relation.FRST}}. We consider several types of relations e.g.,
            fuzzy equivalence, tolerance, and \eqn{T}-similarity relations. These relations can be chosen by
            assigning \code{type.relation}. Additionally, in this function, we provide several options to
            calculate aggregation e.g., triangular norm operators (e.g., \code{"lukasiewicz"}, \code{"min"}, etc)
            and user-defined operators.
	   	 \item Lower and upper approximations: These approximations show to what extent objects can be classified with certainty or not.
          This task has been implemented in

          \code{\link{BC.LU.approximation.FRST}}. There are many approaches available in this package that can be selected by assigning the parameter \code{type.LU}.
          The considered methods are
          implication/t-norm, \eqn{\beta}-precision fuzzy rough sets (\eqn{\beta}-PFRS), vaguely quantified rough sets (VQRS), fuzzy variable precision rough sets (FVPRS), ordered weighted average (OWA),
          soft fuzzy rough sets (SFRS), and robust fuzzy rough sets (RFRS). Furthermore, we provide a facility, which is \code{"custom"}, where users can create their own approximations by
          defining functions to calculate lower and upper approximations. Many options to calculate implicator and triangular norm are also available.
     \item Positive region: It is used to determine the membership degree of each object to the positive region and the degree of dependency.
                  It is implemented in \code{\link{BC.positive.reg.FRST}}.
     \item Discernibility matrix: It is used to construct the decision-relative discernibility matrix. There are some approaches to construct the matrix,
                  e.g., based on standard approach, Gaussian reduction, alpha reduction, and minimal element in discernibility matrix. They have been implemented
                  in \code{\link{BC.discernibility.mat.FRST}}.
   }
   \item Feature selection: According to the output of functions,
      we may divide them into three groups: those that produce a superreduct, a set of reducts, or a single reduct. The following is a description of functions based on their types:
      \itemize{
          \item Feature subset: It refers to methods which produce a superreduct which is not necessarily a reduct. In other words methods in this group
                              might generate just a subset of attributes.
                The following is a complete list of methods considered in this package:
                \itemize{
                 \item positive region based algorithms: It refers to
                       positive regions, as a way to evaluate attributes to be selected. They are implemented in \code{\link{FS.quickreduct.FRST}}.
                       Furthermore, we provide several different measures based on the positive region in this function.
                       All methods included in this part employ the QuickReduct algorithm to obtain selected features.
                       In order to choose a particular algorithm, we need to assign parameter \code{type.method} in \code{\link{FS.quickreduct.FRST}}.
               \item boundary region based algorithm: This algorithm is based on the membership degree to the fuzzy boundary region.
                     This algorithm has been implemented in \code{\link{FS.quickreduct.FRST}}.
              }
              Furthermore, we provide a wrapper function \code{\link{FS.feature.subset.computation}} in order to give a user interface for many methods of RST and FRST.
          \item Reduct: It refers to a method that produces a single decision reduct. We provide one algorithm which is the near-optimal reduction proposed by Zhao et al.
                      It is implemented in \code{\link{FS.nearOpt.fvprs.FRST}}.
               Furthermore, we provide a wrapper function \code{\link{FS.reduct.computation}} in order to provide a user interface toward many methods of RST and FRST.
          \item All reducts:  In order to get all decision reducts, we execute \code{\link{FS.all.reducts.computation}}. However, before doing that, we firstly execute the \code{\link{BC.discernibility.mat.FRST}} function for
                            constructing a decision-relative discernibility matrix.
    }
    The output of the above methods is a class containing a decision reduct/feature subset and other descriptions.
    For generating a new decision table according to the decision reduct, we provide the function \code{\link{SF.applyDecTable}}.

    \item Rule induction: It is a task used to generate
          rules representing knowledge of a decision table. Commonly, this process is called learning phase in machine learning.
          The following methods are considered to generate rules:
          \itemize{
	    		 \item \code{\link{RI.hybridFS.FRST}}: It combines fuzzy-rough rule induction
                     and feature selection.
              \item \code{\link{RI.GFRS.FRST}}: It refers to rule induction based on generalized fuzzy rough sets (GFRS).
          }
          After generating rules, we can use them to predict decision values/classes of new data
          by executing the S3 function \code{\link{predict.RuleSetFRST}}.
    \item Instance selection: The following functions select instances to improve accuracy by
          removing noisy, superfluous or inconsistent ones from training datasets.
          \itemize{
            \item \code{\link{IS.FRIS.FRST}}: It refers to the fuzzy rough instance selection (FRIS). It evaluates the degree of membership to the positive region of each instance.
            		  If an instance's membership degree is less than the threshold, then the instance can be removed.
            \item \code{\link{IS.FRPS.FRST}}: It refers to the fuzzy-rough prototype selection (FRPS). It employs prototype selection (PS) to improve the accuracy of the $k$-nearest neighbor (kNN) method.
          }
      We provide the function \code{\link{SF.applyDecTable}} that is used to generate a new decision table according to the output of instance selection functions.
   \item Fuzzy-rough nearest neighbors: This part provides methods based on nearest neighbors for
         predicting decision values/classes of new datasets. In other words, by supplying a decision table as training data
         we can predict decision values of new data at the same time.
         We have considered the following methods:
         \itemize{
           \item \code{\link{C.FRNN.FRST}}: It refers to the fuzzy-rough nearest neighbors based on Jensen and Cornelis' technique.
           \item \code{\link{C.FRNN.O.FRST}}: It refers to the fuzzy-rough ownership nearest neighbor algorithm based on Sarkar's method.
           \item \code{\link{C.POSNN.FRST}}: The positive region based fuzzy-rough nearest neighbor algorithm based on Verbiest et al's technique.
         }
}
}

Furthermore, we provide an additional feature which is missing value completion. Even though algorithms, included in this feature, are not based on RST and FRST, they will be usefull to do data analysis.
The following is a list of functions implemented for handling missing values in the data preprocessing step:
\itemize{
\item \code{\link{MV.deletionCases}}: it refers to the approach deleting instances.
\item \code{\link{MV.mostCommonValResConcept}}: it refers to the approach based on the most common value or mean of an attribute restricted to a concept.
\item \code{\link{MV.mostCommonVal}}: it refers to the approach replacing missing attribute values by the attribute mean or common values.
\item \code{\link{MV.globalClosestFit}}: it refers to the approach based on the global closest fit approach.
\item \code{\link{MV.conceptClosestFit}}: it refers to the approach based on the concept closest fit approach.
}
Additionally, we provide a wrapper function which is \code{\link{MV.missingValueCompletion}}
in order to give a user interface for the methods.

To get started with the package, the user can have a look at the examples included in
the documentation on each function. Additionally, to show general usage of the package briefly,
we also provide some examples showing general usage in this section.

If you have problems using the package, find a bug, or have suggestions,
please contact the package maintainer by email, instead of writing to the general R lists
or to other internet forums and mailing lists.

There are many demos that ship with the package. To get a list of them, type:

\code{demo()}

Then, to start a demo, type \code{demo(<demo_name_here>)}. All the demos are presented as
R scripts in the package sources in the "demo" subdirectory.

Currently, there are the following demos available:

\itemize{
\item Basic concepts of RST and FRST:

\code{demo(BasicConcept.RST)},
\code{demo(BasicConcept.FRST)},

\code{demo(DiscernibilityMatrix.RST)},
\code{demo(DiscernibilityMatrix.FRST)}.

\item Discretization based on RST:

\code{demo(D.local.discernibility.matrix.RST)},
\code{demo(D.max.discernibility.matrix.RST)},

\code{demo(D.global.discernibility.heuristic.RST)},
\code{demo(D.discretize.quantiles.RST)},

\code{demo(D.discretize.equal.intervals.RST)}.

\item Feature selection based on RST:

\code{demo(FS.permutation.heuristic.reduct.RST)},
\code{demo(FS.quickreduct.RST)},

\code{demo(FS.greedy.heuristic.reduct.RST)},
\code{demo(FS.greedy.heuristic.reduct.RST)}.

\item Feature selection based on FRST:

\code{demo(FS.QuickReduct.FRST.Ex1)},
\code{demo(FS.QuickReduct.FRST.Ex2)},

\code{demo(FS.QuickReduct.FRST.Ex3)},
\code{demo(FS.QuickReduct.FRST.Ex4)},

\code{demo(FS.QuickReduct.FRST.Ex5)},
\code{demo(FS.nearOpt.fvprs.FRST)}.

\item Instance selection based on FRST:

\code{demo(IS.FRIS.FRST)},
\code{demo(IS.FRPS.FRST)}

\item Classification using the Iris dataset:

\code{demo(FRNN.O.iris)},
\code{demo(POSNN.iris)},
\code{demo(FRNN.iris)}.

\item Rule induction based on RST:

\code{demo(RI.indiscernibilityBasedRules.RST)}.

\item Rule induction based on FRST:

\code{demo(RI.classification.FRST)},
\code{demo(RI.regression.FRST)}.

\item Missing value completion:
\code{demo(MV.simpleData)}.
}

Some decision tables have been embedded in this package which can be seen in
\code{\link{RoughSetData}}.
}
\section{Introduction to Rough Set Theory}{


This part attempts to introduce rough set theory (RST) and its application to data analysis. 
While the classical RST proposed by Pawlak in 1982 is explained in detail in this section, 
some recent advancements will be treated in the documentation of the related functions.  

In RST, a data set is represented as a table called an information system \eqn{\mathcal{A} = (U, A)}, where
\eqn{U} is a non-empty set of finite objects known as the universe of discourse (note: it refers to all instances/rows 
in datasets) and \eqn{A} is a non-empty finite set of attributes, such that \eqn{a : U \to V_{a}} for every \eqn{a \in A}. 
The set \eqn{V_{a}} is the set of values that attribute \eqn{a} may take. Information systems that involve a decision attribute, 
containing classes for each object, are called decision systems or decision tables. More formally, it is a pair \eqn{\mathcal{A} = (U, A \cup \{d\})},
where \eqn{d \notin A} is the decision attribute. The elements of \eqn{A} are called conditional attributes. The information system
representing all data in a particular system may contain redundant parts. It could happen because there are the same 
or indiscernible objects or some superfluous attributes. The indiscernibility relation is a binary relation showing the relation between two objects.
This relation is an equivalence relation.  
Let \eqn{\mathcal{A} = (U, A)} be an information system, then for any \eqn{B \subseteq A} there is an equivalence 
relation \eqn{R_B(x,y)}:

\eqn{R_B(x,y)= \{(x,y) \in U^2 | \forall a \in B, a(x) = a(y)\}}

If \eqn{(x,y) \in R_B(x,y)}, then \eqn{x} and \eqn{y} are indiscernible by attributes from \eqn{B}. The equivalence
classes of the \eqn{B}-indiscernibility relation are denoted \eqn{[x]_{B}}. The indiscernibility relation will be further used to define basic concepts of rough
set theory which are lower and upper approximations.

Let \eqn{B \subseteq A} and \eqn{X \subseteq U},
\eqn{X} can be approximated using the information contained within \eqn{B} by constructing 
the \eqn{B}-lower and \eqn{B}-upper approximations of \eqn{X}:

\eqn{R_B \downarrow X = \{ x \in U | [x]_{B} \subseteq X \}}

\eqn{R_B \uparrow X = \{ x \in U | [x]_{B} \cap X \not= \emptyset \}}

The tuple \eqn{\langle R_B \downarrow X, R_B \uparrow X \rangle} is called a rough set.
The objects in \eqn{R_B \downarrow X} mean that they can be with certainty classified as members of \eqn{X} on the basis of knowledge in \eqn{B}, while
the objects in \eqn{R_B \uparrow X} can be only classified as possible members of \eqn{X} on the basis of knowledge in \eqn{B}. 

In a decision system, for \eqn{X} we use decision concepts (equivalence classes of decision attribute) \eqn{[x]_d}. 
We can define \eqn{B}-lower and \eqn{B}-upper approximations as follows.

\eqn{R_B \downarrow [x]_d = \{ x \in U | [x]_{B} \subseteq [x]_d \}}

\eqn{R_B \uparrow [x]_d = \{ x \in U | [x]_{B} \cap [x]_d \not= \emptyset \}}

The positive, negative and boundary of \eqn{B} regions can be defined as:

\eqn{POS_{B} = \bigcup_{x \in U } R_B \downarrow [x]_d}

The boundary region, \eqn{BND_{B}}, is the set of objects that can possibly, but not certainly, be classified.

\eqn{BND_{B} = \bigcup_{x \in U} R_B \uparrow [x]_d - \bigcup_{x \in U} R_B \downarrow [x]_d}

Furthermore, we can calculate the degree of dependency of the decision on a set of attributes. The decision attribute \eqn{d}
depends totally on a set of attributes \eqn{B}, denoted \eqn{B \Rightarrow d},
if all  attribute values from \eqn{d} are uniquely determined by values of attributes from \eqn{B}. It can be defined as follows.
For \eqn{B \subseteq A}, it is said that \eqn{d} depends on \eqn{B} in a degree of dependency \eqn{\gamma_{B} = \frac{|POS_{B}|}{|U|}}. 

A decision reduct is a set \eqn{B \subseteq A} such that \eqn{\gamma_{B} = \gamma_{A}} and \eqn{\gamma_{B'} < \gamma_{B}} for every \eqn{B' \subset B}.
One algorithm to determine all reducts is by constructing the decision-relative discernibility matrix. 
The discernibility matrix \eqn{M(\mathcal{A})} is an \eqn{n \times n} matrix \eqn{(c_{ij})} where

\eqn{c_{ij} = \{a \in A: a(x_i) \neq a(x_j) \}} if \eqn{d(x_i) \neq d(x_j)} and

\eqn{c_{ij} = \oslash} otherwise

The discernibility function \eqn{f_{\mathcal{A}}} for a decision system \eqn{\mathcal{A}} is a boolean function of \eqn{m} boolean variables \eqn{\bar{a}_1, \ldots, \bar{a}_m}
corresponding to the attributes \eqn{a_1, \ldots, a_m} respectively, and defined by

\eqn{f_{\mathcal{A}}(\bar{a_1}, \ldots, \bar{a_m}) = \wedge \{\vee \bar{c}_{ij}: 1 \le j < i \le n, c_{ij} \neq \oslash \}}

where \eqn{\bar{c}_{ij}= \{ \bar{a}: a \in c_{ij}\}}. The decision reducts of \eqn{A} are then the prime implicants of the function \eqn{f_{\mathcal{A}}}. 
The complete explanation of the algorithm can be seen in (Skowron and Rauszer, 1992).

The implementations of the RST concepts can be seen in \code{\link{BC.IND.relation.RST}}, 

\code{\link{BC.LU.approximation.RST}}, \code{\link{BC.positive.reg.RST}}, and 

\code{\link{BC.discernibility.mat.RST}}.
}

\section{Introduction to Fuzzy Rough Set Theory}{


This part introduces briefly fuzzy rough set theory (FRST) and its application to data analysis. 
Since recently there are a lot of FRST variants that have been
proposed by researchers, in this introduction, we only provide some basic concepts of FRST based on (Radzikowska and Kerre, 2002). 

Just like in RST (see \code{\link{Introduction-RoughSets}}),
a data set is represented as a table called an information system \eqn{\mathcal{A} = (U, A)}, where
\eqn{U} is a non-empty set of finite objects as the universe of discourse (note: it refers to all instances/experiments/rows 
in datasets) and \eqn{A} is a non-empty finite set of attributes, such that \eqn{a : U \to V_{a}} for every \eqn{a \in A}. 
The set \eqn{V_{a}} is the set of values that attribute \eqn{a} may take. Information systems that involve a decision attribute, 
containing classes or decision values of each objects, are called decision systems (or said as decision tables). More formally, it is a pair \eqn{\mathcal{A} = (U, A \cup \{d\})},
where \eqn{d \notin A} is the decision attribute. The elements of \eqn{A} are called conditional attributes. However, different from RST, FRST has several ways
to express indiscernibility. 

Fuzzy indiscernibility relation (FIR) is used for any fuzzy relation that determines the degree to which two objects are indiscernible. 
We consider some special cases of FIR.
\itemize{
\item fuzzy tolerance relation: this relation has properties which are reflexive and symmetric where

      reflexive: \eqn{R(x,x) = 1}

      symmetric: \eqn{R(x,y) = R(y,x)}

\item similarity relation (also called fuzzy equivalence relation): this relation has properties not only reflexive and symmetric but also
      transitive defined as

      \eqn{min(R(x,y), R(y,z)) \le R(x,z)}

\item \eqn{\mathcal{T}}-similarity relation (also called fuzzy \eqn{\mathcal{T}}-equivalence relation): this relation is a fuzzy tolerance relation that is also \eqn{\mathcal{T}}-transitive.

      \eqn{\mathcal{T}(R(x,y), R(y,z)) \le R(x,z)}, for a given triangular norm \eqn{\mathcal{T}}. 
}

The following equations are the tolerance relations on a quantitative attribute \eqn{a}, \eqn{R_a}, proposed by (Jensen and Shen, 2009).
\itemize{
\item \code{eq.1}: \eqn{R_a(x,y) = 1 - \frac{|a(x) - a(y)|}{|a_{max} - a_{min}|}}
\item \code{eq.2}: \eqn{R_a(x,y) = exp(-\frac{(a(x) - a(y))^2}{2 \sigma_a^2})}
\item \code{eq.3}: \eqn{R_a(x,y) = max(min(\frac{a(y) - a(x) + \sigma_a}{\sigma_a}, \frac{a(x) - a(y) + \sigma_a}{\sigma_a}), 0)}
}
where \eqn{\sigma_{a}^2} is the variance of feature \eqn{a} and \eqn{a_{min}} and \eqn{a_{max}} are the minimal and maximal values of data supplied by user. 
Additionally, other relations have been implemented in \code{\link{BC.IND.relation.FRST}}

For a qualitative (i.e., nominal) attribute \eqn{a}, the classical manner of discerning objects is used, i.e., \eqn{R_a(x,y) = 1}
if \eqn{a(x) = a(y)} and \eqn{R_a(x,y) = 0}, otherwise. We can then define, for any subset \eqn{B} of \eqn{A}, the fuzzy \eqn{B}-indiscernibility relation by

\eqn{R_B(x,y) = \mathcal{T}(R_a(x,y))},

where \eqn{\mathcal{T}} is a t-norm operator, for instance minimum, product and Lukasiewicz t-norm. 
In general, \eqn{\mathcal{T}} can be replaced by any aggregation operator, like e.g., the average.

In the context of FRST, according to (Radzikowska and Kerre, 2002) lower and upper approximation 
are generalized by means of an implicator \eqn{\mathcal{I}} and a t-norm \eqn{\mathcal{T}}. 
The following are the fuzzy \eqn{B}-lower and \eqn{B}-upper approximations of a fuzzy set \eqn{A} in \eqn{U}

\eqn{(R_B \downarrow A)(y) = inf_{x \in U} \mathcal{I}(R_B(x,y), A(x))}

\eqn{(R_B \uparrow A)(y) = sup_{x \in U} \mathcal{T}(R_B(x,y), A(x))}

The underlying meaning is that \eqn{R_B \downarrow A} is the set of elements \emph{necessarily} satisfying
the concept (strong membership), while \eqn{R_B \uparrow A} is the set of elements \emph{possibly} belonging
to the concept (weak membership). Many other ways to define the approximations can be found in \code{\link{BC.LU.approximation.FRST}}.
Mainly, these were designed to deal with noise in the data and to make the approximations more robust.

Based on fuzzy \eqn{B}-indiscernibility relations, we define the fuzzy \eqn{B}-positive region by, for \eqn{y \in X},

\eqn{POS_B(y) = (\cup_{x \in U} R_B \downarrow R_dx)(y)}

We can define the degree of dependency of \eqn{d} on \eqn{B}, \eqn{\gamma_{B}} by

\eqn{\gamma_{B} = \frac{|POS_{B}|}{|U|} = \frac{\sum_{x \in U} POS_{B}(x)}{|U|}}

A decision reduct is a set \eqn{B \subseteq A} such that \eqn{\gamma_{B} = \gamma_{A}} and \eqn{\gamma_{B'} = \gamma_{B}} for every \eqn{B' \subset B}. 

As we know from rough set concepts (See \code{\link{Introduction-RoughSets}}), we are able to calculate the 
decision reducts by constructing the decision-relative discernibility matrix. Based on (Tsang et al, 2008), the discernibility matrix
can be defined as follows. 
The discernibility matrix is an \eqn{n \times n} matrix \eqn{(c_{ij})} where 
for \eqn{i,j = 1, \ldots, n}
       
1) \eqn{c_{ij}= \{a \in A : 1 - R_{a}(x_i, x_j) \ge	\lambda_i\}} if \eqn{\lambda_j < \lambda_i}.

2) \eqn{c_{ij}={\oslash}}, otherwise. 

with \eqn{\lambda_i = (R_A \downarrow R_{d}x_{i})(x_i)} and \eqn{\lambda_j = (R_A \downarrow R_{d}x_{j})(x_{j})}

Other approaches of discernibility matrix can be read at \code{\link{BC.discernibility.mat.FRST}}.

The other implementations of the FRST concepts can be seen at \code{\link{BC.IND.relation.FRST}}, 

\code{\link{BC.LU.approximation.FRST}}, and \code{\link{BC.positive.reg.FRST}}.
}

\examples{
##############################################################
## A.1 Example: Basic concepts of rough set theory
##############################################################
## Using hiring data set, see RoughSetData
data(RoughSetData)
decision.table <- RoughSetData$hiring.dt

## define considered attributes which are first, second, and
## third attributes
attr.P <- c(1,2,3)

## compute indiscernibility relation
IND <- BC.IND.relation.RST(decision.table, feature.set = attr.P)

## compute lower and upper approximations
roughset <- BC.LU.approximation.RST(decision.table, IND)

## Determine regions
region.RST <- BC.positive.reg.RST(decision.table, roughset)

## The decision-relative discernibility matrix and reduct
disc.mat <- BC.discernibility.mat.RST(decision.table, range.object = NULL)

##############################################################
## A.2 Example: Basic concepts of fuzzy rough set theory
##############################################################
## Using pima7 data set, see RoughSetData
data(RoughSetData)
decision.table <- RoughSetData$pima7.dt

## In this case, let us consider the first and second attributes
conditional.attr <- c(1, 2)

## We are using the "lukasiewicz" t-norm and the "tolerance" relation
## with "eq.1" as fuzzy similarity equation
control.ind <- list(type.aggregation = c("t.tnorm", "lukasiewicz"),
                    type.relation = c("tolerance", "eq.1"))

## Compute fuzzy indiscernibility relation
IND.condAttr <- BC.IND.relation.FRST(decision.table, attributes = conditional.attr,
                            control = control.ind)

## Compute fuzzy lower and upper approximation using type.LU : "implicator.tnorm"
## Define index of decision attribute
decision.attr = c(9)

## Compute fuzzy indiscernibility relation of decision attribute
## We are using "crisp" for type of aggregation and type of relation
control.dec <- list(type.aggregation = c("crisp"), type.relation = "crisp")

IND.decAttr <- BC.IND.relation.FRST(decision.table, attributes = decision.attr,
                            control = control.dec)

## Define control parameter containing type of implicator and t-norm
control <- list(t.implicator = "lukasiewicz", t.tnorm = "lukasiewicz")

## Compute fuzzy lower and upper approximation
FRST.LU <- BC.LU.approximation.FRST(decision.table, IND.condAttr, IND.decAttr,
              type.LU = "implicator.tnorm", control = control)

## Determine fuzzy positive region and its degree of dependency
fuzzy.region <- BC.positive.reg.FRST(decision.table, FRST.LU)

###############################################################
## B Example : Data analysis based on RST and FRST
## In this example, we are using wine dataset for both RST and FRST
###############################################################
## Load the data
\dontrun{data(RoughSetData)
dataset <- RoughSetData$wine.dt

## Shuffle the data with set.seed
set.seed(5)
dt.Shuffled <- dataset[sample(nrow(dataset)),]

## Split the data into training and testing
idx <- round(0.8 * nrow(dt.Shuffled))
  wine.tra <-SF.asDecisionTable(dt.Shuffled[1:idx,],
decision.attr = 14, indx.nominal = 14)
  wine.tst <- SF.asDecisionTable(dt.Shuffled[
 (idx+1):nrow(dt.Shuffled), -ncol(dt.Shuffled)])

## DISCRETIZATION
cut.values <- D.discretization.RST(wine.tra,
type.method = "global.discernibility")
d.tra <- SF.applyDecTable(wine.tra, cut.values)
d.tst <- SF.applyDecTable(wine.tst, cut.values)

## FEATURE SELECTION
red.rst <- FS.feature.subset.computation(d.tra,
  method="quickreduct.rst")
fs.tra <- SF.applyDecTable(d.tra, red.rst)

## RULE INDUCTION
rules <- RI.indiscernibilityBasedRules.RST(d.tra,
  red.rst)

## predicting newdata
pred.vals <- predict(rules, d.tst)

#################################################
## Examples: Data analysis using the wine dataset
## 2. Learning and prediction using FRST
#################################################

## FEATURE SELECTION
reduct <- FS.feature.subset.computation(wine.tra,
 method = "quickreduct.frst")

## generate new decision tables
wine.tra.fs <- SF.applyDecTable(wine.tra, reduct)
wine.tst.fs <- SF.applyDecTable(wine.tst, reduct)

## INSTANCE SELECTION
indx <- IS.FRIS.FRST(wine.tra.fs,
 control = list(threshold.tau = 0.2, alpha = 1))

## generate a new decision table
wine.tra.is <- SF.applyDecTable(wine.tra.fs, indx)

## RULE INDUCTION (Rule-based classifiers)
control.ri <- list(
 type.aggregation = c("t.tnorm", "lukasiewicz"),
 type.relation = c("tolerance", "eq.3"),
 t.implicator = "kleene_dienes")

decRules.hybrid <- RI.hybridFS.FRST(wine.tra.is,
  control.ri)

## predicting newdata
predValues.hybrid <- predict(decRules.hybrid,
  wine.tst.fs)

#################################################
## Examples: Data analysis using the wine dataset
## 3. Prediction using fuzzy nearest neighbor classifiers
#################################################

## using FRNN.O
control.frnn.o <- list(m = 2,
  type.membership = "gradual")

predValues.frnn.o <- C.FRNN.O.FRST(wine.tra.is,
  newdata = wine.tst.fs, control = control.frnn.o)

## Using FRNN
control.frnn <- list(type.LU = "implicator.tnorm",k=20,
  type.aggregation = c("t.tnorm", "lukasiewicz"),
  type.relation = c("tolerance", "eq.1"),
  t.implicator = "lukasiewicz")

predValues.frnn <- C.FRNN.FRST(wine.tra.is,
  newdata = wine.tst.fs, control = control.frnn)

## calculating error
real.val <- dt.Shuffled[(idx+1):nrow(dt.Shuffled),
  ncol(dt.Shuffled), drop = FALSE]

err.1 <- 100*sum(pred.vals!=real.val)/nrow(pred.vals)
err.2 <- 100*sum(predValues.hybrid!=real.val)/
  nrow(predValues.hybrid)
err.3 <- 100*sum(predValues.frnn.o!=real.val)/
  nrow(predValues.frnn.o)
err.4 <- 100*sum(predValues.frnn!=real.val)/
  nrow(predValues.frnn)

cat("The percentage error = ", err.1, "\n")
cat("The percentage error = ", err.2, "\n")
cat("The percentage error = ", err.3, "\n")
cat("The percentage error = ", err.4, "\n")}
}
\references{
D. Dubois and H. Prade, "Rough Fuzzy Sets and Fuzzy Rough Sets",
International Journal of General Systems, vol. 17, p. 91 - 209 (1990).

\emph{General references}

L.A. Zadeh, "Fuzzy Sets",
Information and Control, vol. 8, p. 338 - 353 (1965).

Z. Pawlak, "Rough Sets",
International Journal of Computer and Information System,
vol. 11, no. 5, p. 341 - 356 (1982).

Z. Pawlak, "Rough Sets: Theoretical Aspects of Reasoning About Data, System Theory, Knowledge Engineering and Problem Solving",
vol. 9, Kluwer Academic Publishers, Dordrecht, Netherlands (1991).

\emph{Introduction to Rough Set Theory  references}

A. Skowron and C. Rauszer,  
"The Discernibility Matrices and Functions in Information Systems", 
in: R. Slowinski (Ed.), Intelligent Decision Support: Handbook of Applications and
Advances of Rough Sets Theory, Kluwer Academic Publishers, Dordrecht, Netherland,  
p. 331 - 362 (1992).

Z. Pawlak, "Rough Sets", 
International Journal of Computer and Information System, 
vol. 11, no.5, p. 341 - 356 (1982).

Z. Pawlak, "Rough Sets: Theoretical Aspects of Reasoning about Data, System Theory, Knowledge Engineering and Problem Solving",
vol. 9, Kluwer Academic Publishers, Dordrecht, Netherlands (1991). 

\emph{Introduction to Fuzzy Rough Set Theory references}

A. M. Radzikowska and E. E. Kerre, "A Comparative Study of Fuzzy Rough Sets", 
Fuzzy Sets and Systems, vol. 126, p. 137 - 156 (2002). 

D. Dubois and H. Prade, "Rough Fuzzy Sets and Fuzzy Rough Sets",
International Journal of General Systems, vol. 17, p. 91 - 209 (1990).

E. C. C. Tsang, D. G. Chen, D. S. Yeung, X. Z. Wang, and J. W. T. Lee, 
"Attributes Reduction Using Fuzzy Rough Sets", IEEE Trans. Fuzzy Syst., 
vol. 16, no. 5, p. 1130 - 1141 (2008).

L. A. Zadeh, "Fuzzy Sets",
Information and Control, vol. 8, p. 338 - 353 (1965).

R. Jensen and Q. Shen,  
"New Approaches to Fuzzy-Rough Feature Selection", 
IEEE Trans. on Fuzzy Systems, vol. 19, no. 4,
p. 824 - 838 (2009).

Z. Pawlak, "Rough Sets", International Journal of Computer and Information Sciences, 
vol. 11, no. 5, p. 341 - 356 (1982).
}
\author{
Lala Septem Riza \email{lala.s.riza@decsai.ugr.es},

Andrzej Janusz \email{andrzejanusz@gmail.com},

Chris Cornelis \email{chriscornelis@decsai.ugr.es},

Francisco Herrera \email{herrera@decsai.ugr.es},

Dominik Slezak \email{slezak@mimuw.edu.pl},

and Jose Manuel Benitez \email{j.m.benitez@decsai.ugr.es}

DiCITS Lab, SCI2S group, CITIC-UGR, DECSAI, University of Granada,

\url{https://sci2s.ugr.es}

Institute of Mathematics, University of Warsaw.
}
