% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SESraster.R
\name{SESraster}
\alias{SESraster}
\title{Standardized effect sizes for SpatRaster objects}
\usage{
SESraster(
  x,
  FUN = NULL,
  FUN_args = list(),
  spat_alg = NULL,
  spat_alg_args = list(),
  Fa_sample = NULL,
  Fa_alg = NULL,
  Fa_alg_args = list(),
  aleats = 10,
  cores = 1,
  filename = "",
  overwrite = FALSE,
  force_wr_aleat_file = FALSE,
  ...
)
}
\arguments{
\item{x}{SpatRaster. A SpatRaster containing presence-absence data (0 or 1)
for a set of species.}

\item{FUN}{The function to be applied. It must work with SpatRaster objects.
See examples.}

\item{FUN_args}{Named list of arguments passed to the FUN}

\item{spat_alg}{A function with the algorithm implementing the desired
randomization method. It must work with SpatRaster objects. See examples.
Example of functions that work are: \code{\link{bootspat_naive}},
\code{\link{bootspat_str}}, \code{\link{bootspat_ff}}.}

\item{spat_alg_args}{List of arguments passed to the randomization method
chosen in 'spat_alg'. See \code{\link{bootspat_naive}}, \code{\link{bootspat_str}},
\code{\link{bootspat_ff}}}

\item{Fa_sample}{Named list of length 1 with a FUN argument (e.g. a vector)
to be randomized}

\item{Fa_alg}{function to randomize any non spatial argument to be passed
to 'FUN'.}

\item{Fa_alg_args}{Named list of arguments passed to the function in 'Fa_alg'}

\item{aleats}{positive integer. A positive integer indicating how many times
the calculation should be repeated.}

\item{cores}{positive integer. If \code{cores > 1}, a 'parallel' package cluster with that many cores is created and used. You can also supply a cluster object. Ignored for functions that are implemented by terra in C++ (see under fun)}

\item{filename}{character. Output filename}

\item{overwrite}{logical. If \code{TRUE}, \code{filename} is overwritten}

\item{force_wr_aleat_file}{logical. Force writing bootstrapped rasters, even if
files fit in memory. Mostly used for internal test units.}

\item{...}{additional arguments passed to 'terra::app()' function.}
}
\value{
SpatRaster. The function returns the observed metric, the mean of the
simulations calculated over n=aleats times, the standard deviation of the
simulations, the standardized effect size (SES) for the metric defined in FUN,
and p values for the upper and lower tails.
}
\description{
Calculates the standardized effect sizes using a custom function
and a null model algorithm.
}
\details{
Perform n=aleats spatial randomizations based on the randomization
method defined in 'spat_alg' argument and calculates the metric
defined in 'FUN' argument. The function (FUN) to calculate the desired metric
must work with any of \link[terra]{app}, \link[terra]{focal},
\link[terra]{focal3D} family of functions.
}
\examples{
library(SESraster)
library(terra)
r <- load_ext_data()
appmean <- function(x, ...){
                      terra::app(x, "mean", ...)
                    }
ses <- SESraster(r, FUN=appmean, spat_alg = "bootspat_naive", spat_alg_args=list(random="species"),
                 aleats = 4)
plot(ses)
ses <- SESraster(r, FUN=appmean, spat_alg = "bootspat_naive", spat_alg_args=list(random="site"),
                 aleats = 4)
plot(ses)

## example of how to use 'FUN_args'
r[7][1] <- NA
plot(r)
set.seed(10)
sesNA <- SESraster(r, FUN=appmean, FUN_args = list(na.rm = FALSE),
                 spat_alg = "bootspat_naive", spat_alg_args=list(random = "species"),
                 aleats = 4)
plot(sesNA)

set.seed(10)
ses <- SESraster(r, FUN=appmean, FUN_args = list(na.rm = TRUE),
                spat_alg = "bootspat_naive", spat_alg_args=list(random = "species"),
                 aleats = 4)
plot(ses)

## example with 'Fa_alg'
appsv <- function(x, lyrv, na.rm = FALSE, ...){
                      sumw <- function(x, lyrv, na.rm, ...){
                            ifelse(all(is.na(x)), NA,
                                    sum(x*lyrv, na.rm=na.rm, ...))
                      }
                      stats::setNames(terra::app(x, sumw, lyrv = lyrv, na.rm=na.rm, ...), "sumw")
                    }
set.seed(10)
ses <- SESraster(r, FUN=appsv,
                 FUN_args = list(lyrv = seq_len(nlyr(r)), na.rm = TRUE),
                    Fa_sample = "lyrv",
                    Fa_alg = "sample", Fa_alg_args = list(replace=FALSE),
                    aleats = 4)
plot(ses)

set.seed(10)
ses <- SESraster(r, FUN=appsv,
                 FUN_args = list(lyrv = seq_len(nlyr(r)), na.rm = TRUE),
                    Fa_sample = "lyrv",
                    Fa_alg = "sample", Fa_alg_args = list(replace=TRUE),
                    aleats = 4)
plot(ses)

}
\references{
Gotelli 2000
}
\seealso{
\code{\link{bootspat_str}}, \code{\link{bootspat_naive}},
\code{\link{bootspat_ff}}, \code{\link{algorithm_metrics}}
}
\author{
Neander M. Heming and Gabriela Alves-Ferreira
}
