% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SNSeg_HD.R
\name{SNSeg_HD}
\alias{SNSeg_HD}
\title{Self-normalization (SN) based change points estimation for high dimensional
time series for changes in high-dimensional means (SNHD).}
\usage{
SNSeg_HD(
  ts,
  confidence = 0.9,
  grid_size_scale = 0.05,
  grid_size = NULL,
  plot_SN = FALSE,
  est_cp_loc = TRUE,
  ts_index = c(1:5)
)
}
\arguments{
\item{ts}{A high-dimensional time series represented as a matrix with p
columns, where each column is a univariate time series. The dimension p for
ts should be at least 10.}

\item{confidence}{Confidence level of SN tests as a numeric value. Available
choices of confidence levels contain 0.9, 0.95, 0.99, 0.995 and 0.999. The
default is set to 0.9.}

\item{grid_size_scale}{numeric value of the trimming parameter and only in
use if grid_size = NULL. Users are allowed to choose any grid_size_scale
between 0.05 and 0.5. A warning will be given if it is outside the
range.}

\item{grid_size}{Local window size h to compute the critical value for SN
test. Since grid_size = n*grid_size_scale, where n is the length of time
series, this function will compute the grid_size_scale by diving n from
grid_size when it is not NULL.}

\item{plot_SN}{Boolean value to plot the time series or not.
The default setting is FALSE.}

\item{est_cp_loc}{Boolean value to plot a red solid vertical line for
estimated change-point locations if est_cp_loc = TRUE.}

\item{ts_index}{The index number(s) of the univariate time series to be plotted.
Users should enter a positive integer or a vector of positive integers that are
no greater than the dimension of the input time series. The default is the
first 5 time series, i.e., \code{ts_index = c(1:5)}.}
}
\value{
SNSeg_HD returns an S3 object of class "SNSeg_HD" including the time
series, the local window size to cover a change point, the estimated change-point
locations, the confidence level and the critical value of the SN test. It also
generates time series segmentation plot when \code{plot_SN = TRUE}.
\describe{
\item{\code{ts}}{A numeric matrix of the input time series.}
\item{\code{grid_size}}{A numeric value of the window size.}
\item{\code{SN_sweep_result}}{A list of n matrices where each matrix
consists of four columns: (1) SN-based test statistic for each change-point
location (2) Change-point location  (3) Lower bound of the window h and
(4) Upper bound of the window h.}
\item{\code{est_cp}}{A vector containing the locations of the estimated
change-points.}
\item{\code{confidence}}{Confidence level of SN test as a numeric value.}
\item{\code{critical_value}}{Critical value of the SN-based test statistic.}
}

Users can apply the functions \code{summary.SN} to compute the parameter estimate
of each segment separated by the detected change-points. An additional function
\code{plot.SN} can be used to plot the time series with estimated change-points.
Users can set the option \code{plot_SN = TRUE} or use the function \code{plot.SN}
to plot the time series.

It deserves to note that some change-points could be missing due to the constraint
on \code{grid_size_scale} or related \code{grid_size} that \code{grid_size_scale}
has a minimum value of 0.05. Therefore, SNCP claims no change-points within the
first n\emph{\code{grid_size_scale} or the last n}\code{grid_size_scale} time points.
This is a limitation of the function \code{SNSeg_HD}.

For more examples of \code{SNSeg_HD} see the help vignette:
\code{vignette("SNSeg", package = "SNSeg")}
}
\description{
The function \code{SNSeg_HD} is a SNHD change point
estimation procedure.
}
\examples{
\donttest{
n <- 500
p <- 50
nocp <- 5
cp_sets <- round(seq(0,nocp+1,1)/(nocp+1)*n)
num_entry <- 5
kappa <- sqrt(4/5)
mean_shift <- rep(c(0,kappa),100)[1:(length(cp_sets)-1)]
set.seed(1)
ts <- matrix(rnorm(n*p,0,1),n,p)
no_seg <- length(cp_sets)-1
for(index in 1:no_seg){
  tau1 <- cp_sets[index]+1
  tau2 <- cp_sets[index+1]
  ts[tau1:tau2,1:num_entry] <- ts[tau1:tau2,1:num_entry] +
    mean_shift[index]
}

# grid_size defined
result <- SNSeg_HD(ts, confidence = 0.9, grid_size_scale  = 0.05,
                   grid_size = 40)
# Estimated change-point locations
result$est_cp

# For more examples, please run the following command:
# vignette("SNSeg", package = "SNSeg")
}

}
