#' Plot Observed Data and BSTS Forecast
#'
#' Creates a plot of observed data, forecasted values, and confidence intervals.
#'
#' @param forecast A matrix of BSTS forecast samples.
#' @param data_train Numeric vector, training data.
#' @param data_test Numeric vector, test data.
#' @param time Numeric vector, representing time indices.
#' @param quant_high Numeric, upper quantile for confidence interval.
#' @param quant_low Numeric, lower quantile for confidence interval.
#' @param observed_col Character, color for observed data.
#' @param forecast_col Character, color for forecasted data.
#' @param title Character, title of the plot.
#' @return A ggplot2 object.
#' @importFrom ggplot2 ggplot aes geom_line geom_ribbon scale_color_manual scale_fill_manual theme_light labs theme
#' @importFrom stats quantile
#' @export
#'
plot_forecast <- function(forecast, data_train, data_test, time, quant_high, quant_low,
                          observed_col, forecast_col, title){

  # helper to turn non-finite into NA so geoms can quietly skip them
  clean_num <- function(x) replace(x, !is.finite(x), NA_real_)

  Observed    <- clean_num(c(data_train, rep(NA, length(data_test))))
  Future      <- clean_num(c(rep(NA, (length(data_train) - 1)),
                             data_train[length(data_train)], data_test))
  AvgForecast <- clean_num(c(rep(NA, (length(data_train) - 1)),
                             data_train[length(data_train)], colMeans(forecast)))
  QuantHigh   <- clean_num(c(rep(NA, length(data_train)),
                             apply(forecast, 2, quantile, probs = quant_high)))
  QuantLow    <- clean_num(c(rep(NA, length(data_train)),
                             apply(forecast, 2, quantile, probs = quant_low)))
  Time        <- as.numeric(time)

  # make sure lengths line up (avoid silent recycling)
  n <- length(Time)
  stopifnot(all(lengths(list(Observed, Future, AvgForecast, QuantHigh, QuantLow)) == n))

  res_data <- data.frame(Time, Observed, Future, AvgForecast, QuantHigh, QuantLow)

  ts_plot <- ggplot(res_data, aes(x = Time)) +
    # observed
    geom_line(aes(y = Observed, color = "Observed Data"), size = 0.8, na.rm = TRUE) +
    # last obs + test (dotted)
    geom_line(aes(y = Future, color = "Observed Forecast"), linetype = "dotted", size = 1, na.rm = TRUE) +
    # average forecast
    geom_line(aes(y = AvgForecast, color = "Average Forecast"), size = 1, na.rm = TRUE) +
    # interval ribbon
    geom_ribbon(aes(ymin = QuantLow, ymax = QuantHigh, fill = "Confidence Interval"), alpha = 0.45, na.rm = TRUE) +
    # interval bounds (thin lines)
    geom_line(aes(y = QuantLow,  color = "Confidence Interval"), size = 0.1, na.rm = TRUE) +
    geom_line(aes(y = QuantHigh, color = "Confidence Interval"), size = 0.1, na.rm = TRUE) +

    labs(title = title, x = "Time", y = "Crop yields") +

    # IMPORTANT: include a color for "Confidence Interval" or those layers get dropped
    scale_color_manual(
      name = "Legend",
      values = c(
        "Observed Data"        = observed_col,
        "Observed Forecast"    = observed_col,
        "Average Forecast"     = forecast_col,
        "Confidence Interval"  = forecast_col  # added
      )
    ) +
    scale_fill_manual(name = "", values = c("Confidence Interval" = forecast_col)) +

    theme_light() +
    theme(legend.position = "none")

  return(ts_plot)
}


