% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hpwd}
\alias{hpwd}
\title{Heuristic Product Within Distance (Spatially Balanced Sampling Design)}
\usage{
hpwd(dis, n, beta = 10, nrepl = 1L)
}
\arguments{
\item{dis}{A distance matrix NxN that specifies how far all the pairs
of units in the population are.}

\item{n}{Sample size.}

\item{beta}{Parameter \eqn{\beta} for the algorithm. The higher
\eqn{\beta} is, the more the sample is going to be spread (default = 10).}

\item{nrepl}{Number of samples to draw (default = 1).}
}
\value{
Returns a matrix \code{nrepl} x \code{n}, which contains the
\code{nrepl} selected samples, each of them stored in a row. In particular,
the i-th row contains all labels of units selected in the i-th sample.
}
\description{
Selects spatially balanced samples through the use of
Heuristic Product Within Distance design (HPWD). To have constant inclusion
probabilities \eqn{\pi_{i}=n/N}, where \eqn{n} is sample size
and \eqn{N} is population size, the distance matrix has to be standardized
with function \code{\link{stprod}}.
}
\details{
The HPWD design generates samples approximately with the same
probabilities of the \code{\link{pwd}} but with a significantly smaller
number of steps. In fact, this algorithm randomly selects a sample of size
\eqn{n} exactly with \eqn{n} steps, updating at each step the selection
probability of not-selected units, depending on their distance from the
units that were already selected in the previous steps.
}
\examples{
# Example 1
# Draw 1 sample of dimension 10 without constant inclusion probabilities
dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
s <- hpwd(dis = dis, n = 10) # drawn sample
\donttest{
# Example 2
# Draw 1 sample of dimension 15 with constant inclusion probabilities
# equal to n/N, with N = population size
dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
con <- rep(1, nrow(dis)) # vector of constraints
stand_dist <- stprod(mat = dis, con = con) # standardized matrix
s <- hpwd(dis = stand_dist$mat, n = 15) # drawn sample

# Example 3
# Draw 2 samples of dimension 15 with constant inclusion probabilities
# equal to n/N, with N = population size, and an increased level of spread, beta = 20
dis <- as.matrix(dist(cbind(lucas_abruzzo$x, lucas_abruzzo$y))) # distance matrix
con <- rep(0, nrow(dis)) # vector of constraints
stand_dist <- stprod(mat = dis, con = con) # standardized matrix
s <- hpwd(dis = stand_dist$mat, n = 15, beta = 20, nrepl = 2) # drawn samples
}
}
\references{
Benedetti R, Piersimoni F (2017). A spatially balanced design with
probability function proportional to the within sample distance.
\emph{Biometrical Journal}, \strong{59}(5), 1067-1084.
\doi{10.1002/bimj.201600194}

Benedetti R, Piersimoni F (2017). Fast Selection of Spatially Balanced Samples. \emph{arXiv}.
\url{https://arxiv.org/abs/1710.09116}
}
