% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{computeNormalizedLife}
\alias{computeNormalizedLife}
\title{A Vector Summary of the Normalized Life Curve}

\description{
For a given persistence diagram \eqn{D=\{(b_i,d_i)\}_{i=1}^N} (corresponding to a specified homological dimension), \code{computeNormalizedLife()} vectorizes the normalized life curve
\deqn{sl(t)=\sum_{i=1}^N \frac{d_i-b_i}{L}\bold{1}_{[b_i,d_i)}(t),}
where \eqn{L=\sum_{i=1}^N (d_i-b_i)}, based on a scale sequence \code{scaleSeq}. The evaluation method depends on the argument \code{evaluate}. Points in \eqn{D} with infinite death values are ignored.
}

\usage{
computeNormalizedLife(D, homDim, scaleSeq, evaluate = "intervals")
}

\arguments{
\item{D}{a persistence diagram: a matrix with three columns containing the homological dimension, birth and death values respectively.}

\item{homDim}{the homological dimension (0 for \eqn{H_0}, 1 for \eqn{H_1}, etc.). Rows in \code{D} are filtered based on this value.}

\item{scaleSeq}{a numeric vector of increasing scale values used for vectorization.}

\item{evaluate}{a character string indicating the evaluation method. Must be either \code{"intervals"} (default) or \code{"points"}.}

}

\details{
The function extracts rows from \code{D} where the first column equals \code{homDim}, and computes values based on the filtered data and \code{scaleSeq}. If \code{D} does not contain any points corresponding to \code{homDim}, a vector of zeros is returned.

}
\value{
A numeric vector containing elements computed using \code{scaleSeq}=\eqn{\{t_1,t_2,\ldots,t_n\}} according to the method specified by \code{evaluate}.

\itemize{
  \item \code{"intervals"}: Computes average values of the normalized life curve over intervals defined by consecutive elements in \code{scaleSeq}:

  \deqn{\Big(\frac{1}{\Delta t_1}\int_{t_1}^{t_2}sl(t)dt,\frac{1}{\Delta t_2}\int_{t_2}^{t_3}sl(t)dt,\ldots,\frac{1}{\Delta t_{n-1}}\int_{t_{n-1}}^{t_n}sl(t)dt\Big)\in\mathbb{R}^{n-1},}
where \eqn{\Delta t_k=t_{k+1}-t_k}.

  \item \code{"points"}: Computes values of the normalized life curve at each point in \code{scaleSeq}:

  \deqn{(sl(t_1),sl(t_2),\ldots,sl(t_n))\in\mathbb{R}^n.}
}
}

\author{Umar Islambekov}

\references{
Chung, Y. M., & Lawson, A. (2022). Persistence curves: A canonical framework for summarizing persistence diagrams. Advances in Computational Mathematics, 48(1), 1-42.
}

\examples{
N <- 100 # The number of points to sample

set.seed(123) # Set a random seed for reproducibility

# Sample N points uniformly from the unit circle and add Gaussian noise
theta <- runif(N, min = 0, max = 2 * pi)
X <- cbind(cos(theta), sin(theta)) + rnorm(2 * N, mean = 0, sd = 0.2)

# Compute the persistence diagram using the Rips filtration built on top of X
# The 'threshold' parameter specifies the maximum distance for building simplices
D <- TDAstats::calculate_homology(X, threshold = 2)

scaleSeq = seq(0, 2, length.out = 11) # A sequence of scale values

# Compute a vector summary of the normalized life curve for homological dimension H_0
computeNormalizedLife(D, homDim = 0, scaleSeq)

# Compute a vector summary of the normalized life curve for homological dimension H_1
computeNormalizedLife(D, homDim = 1, scaleSeq)
}
