% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GraphicalTesting.R
\docType{class}
\name{GraphicalTesting}
\alias{GraphicalTesting}
\title{Class of GraphicalTesting}
\description{
Perform graphical testing under group sequential design for one or multiple
endpoints. See Maurer & Bretz (2013).
}
\examples{

## Example 1
## dry-run to study the behavior of a graph
## without group sequential design
if(interactive()){
eps <- .01
alpha <- c(.01, .04, 0, 0, 0)
transition <- matrix(c(
  0, 0, 0, 0, 1,
  0, 0, .75, 0, .25,
  0, 1/2-eps/2, 0, eps, 1/2-eps/2,
  0, 0, 0, 0, 0,
  0, 1/2, 1/2, 0, 0
), nrow = 5, byrow = TRUE)

## dummy can be anything, we don't actually use it
asf <- rep('asOF', 5)
## dummy can be anything, we don't actually use it
max_info <- c(300, 1100, 1100, 1100, 500)

hs <- c('H1: UPCR IgA', 'H2: eGFR GN', 'H3: eGFR GN 10wk', 'H5: 2nd Endpoints', 'H4: eGFR IgA')

## initialize an object
gt <- GraphicalTesting$new(alpha, transition, asf, max_info, hs)
print(gt)

## reject hypotheses based on customized order
## to understand the behavior of a testing strategy
## Any other rejection order is possible
gt$reject_a_hypothesis('H1: UPCR IgA')
print(gt)

gt$reject_a_hypothesis('H2: eGFR GN')
print(gt)

gt$reject_a_hypothesis('H4: eGFR IgA')
print(gt)

gt$reject_a_hypothesis('H3: eGFR GN 10wk')
print(gt)

gt$reset()
}

## Example 2
## Example modified from vignettes in gMCPLite:
## Graphical testing for group sequential design
if(interactive()){
## initial alpha split to each of the hypotheses
alpha <- c(.01, .01, .004, .0, .0005, .0005)

## transition matrix of the initial graph
transition <- matrix(c(
  0, 1, 0, 0, 0, 0,
  0, 0, .5, .5, 0, 0,
  0, 0, 0, 1, 0, 0,
  0, 0, 0, 0, .5, .5,
  0, 0, 0, 0, 0, 1,
  .5, .5, 0, 0, 0, 0
), nrow = 6, byrow = TRUE)

## alpha spending functions per hypothesis
asf <- c('asUser', 'asOF', 'asUser', 'asOF', 'asOF', 'asOF')

## planned maximum number of events per hypothesis
max_info <- c(295, 800, 310, 750, 500, 1100)

## name of hypotheses
hs <- c('H1: OS sub',
        'H2: OS all',
        'H3: PFS sub',
        'H4: PFS all',
        'H5: ORR sub',
        'H6: ORR all')

gt <- GraphicalTesting$new(alpha, transition, asf, max_info, hs)

## print initial graph
gt

## nominal p-values at each stage
## Note: p-values with same order are calculated with the same locked data
## Note: alpha_spent is only specified for hypotheses using custom alpha
##       spending function "asUser"
stats <-
  data.frame(
    order = c(1:3, 1:3, 1:2, 1:2, 1, 1),
    hypotheses = c(rep('H1: OS sub', 3), rep('H2: OS all', 3),
                   rep('H3: PFS sub', 2), rep('H4: PFS all', 2),
                   'H5: ORR sub', 'H6: ORR all'),
    p = c(.03, .0001, .000001, .2, .15, .1, .2, .001, .3, .2, .00001, .1),
    info = c(185, 245, 295, 529, 700, 800, 265, 310, 675, 750, 490, 990),
    is_final = c(F, F, T, F, F, T, F, T, F, T, T, T),
    max_info = c(rep(295, 3), rep(800, 3), rep(310, 2), rep(750, 2), 490, 990),
    alpha_spent = c(c(.1, .4, 1), rep(NA, 3), c(.3, 1), rep(NA, 2), NA, NA)
  )

## test the p-values from the first analysis, plot the updated graph
gt$test(stats \%>\% dplyr::filter(order==1))

## test the p-values from the second analysis, plot the updated graph
gt$test(stats \%>\% dplyr::filter(order==2))

## test the p-values from the third analysis, plot the updated graph
## because no futher test would be done, displayed results are final
gt$test(stats \%>\% dplyr::filter(order==3))


## plot the final status of the graph
print(gt, trajectory = FALSE)

## you can get final testing results as follow
gt$get_current_testing_results()

## if you want to see step-by-step details
print(gt$get_trajectory())

## equivalently, you can call gt$test(stats) for only once to get same results.
gt$reset()
gt$test(stats)

## if you only want to get the final testing results
gt$get_current_decision()
}

}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-GraphicalTestingProcedure-new}{\code{GraphicalTesting$new()}}
\item \href{#method-GraphicalTestingProcedure-reset}{\code{GraphicalTesting$reset()}}
\item \href{#method-GraphicalTestingProcedure-is_valid_hid}{\code{GraphicalTesting$is_valid_hid()}}
\item \href{#method-GraphicalTestingProcedure-get_hypothesis_name}{\code{GraphicalTesting$get_hypothesis_name()}}
\item \href{#method-GraphicalTestingProcedure-get_weight}{\code{GraphicalTesting$get_weight()}}
\item \href{#method-GraphicalTestingProcedure-set_weight}{\code{GraphicalTesting$set_weight()}}
\item \href{#method-GraphicalTestingProcedure-get_alpha}{\code{GraphicalTesting$get_alpha()}}
\item \href{#method-GraphicalTestingProcedure-set_alpha}{\code{GraphicalTesting$set_alpha()}}
\item \href{#method-GraphicalTestingProcedure-get_hypotheses_ids}{\code{GraphicalTesting$get_hypotheses_ids()}}
\item \href{#method-GraphicalTestingProcedure-get_number_hypotheses}{\code{GraphicalTesting$get_number_hypotheses()}}
\item \href{#method-GraphicalTestingProcedure-get_hids_not_in_graph}{\code{GraphicalTesting$get_hids_not_in_graph()}}
\item \href{#method-GraphicalTestingProcedure-get_testable_hypotheses}{\code{GraphicalTesting$get_testable_hypotheses()}}
\item \href{#method-GraphicalTestingProcedure-has_testable_hypotheses}{\code{GraphicalTesting$has_testable_hypotheses()}}
\item \href{#method-GraphicalTestingProcedure-is_in_graph}{\code{GraphicalTesting$is_in_graph()}}
\item \href{#method-GraphicalTestingProcedure-is_testable}{\code{GraphicalTesting$is_testable()}}
\item \href{#method-GraphicalTestingProcedure-get_hid}{\code{GraphicalTesting$get_hid()}}
\item \href{#method-GraphicalTestingProcedure-reject_a_hypothesis}{\code{GraphicalTesting$reject_a_hypothesis()}}
\item \href{#method-GraphicalTestingProcedure-set_trajectory}{\code{GraphicalTesting$set_trajectory()}}
\item \href{#method-GraphicalTestingProcedure-get_trajectory}{\code{GraphicalTesting$get_trajectory()}}
\item \href{#method-GraphicalTestingProcedure-test_hypotheses}{\code{GraphicalTesting$test_hypotheses()}}
\item \href{#method-GraphicalTestingProcedure-test}{\code{GraphicalTesting$test()}}
\item \href{#method-GraphicalTestingProcedure-get_current_testing_results}{\code{GraphicalTesting$get_current_testing_results()}}
\item \href{#method-GraphicalTestingProcedure-get_current_decision}{\code{GraphicalTesting$get_current_decision()}}
\item \href{#method-GraphicalTestingProcedure-print}{\code{GraphicalTesting$print()}}
\item \href{#method-GraphicalTestingProcedure-clone}{\code{GraphicalTesting$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-new"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-new}{}}}
\subsection{Method \code{new()}}{
Initialize an object for graphical testing procedure.
Group sequential design is also supported.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$new(
  alpha,
  transition,
  alpha_spending,
  planned_max_info,
  hypotheses = NULL,
  silent = FALSE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{alpha}}{initial alpha allocated to each of the hypotheses.}

\item{\code{transition}}{matrix of transition weights. Its diagonals should
be all 0s. The row sums should be 1s (for better power) or
0s (if no outbound edge from a node).}

\item{\code{alpha_spending}}{character vector of same length of \code{alpha}.
Currently it supports \code{'asP'}, \code{'asOF'}, and \code{'asUser'}.}

\item{\code{planned_max_info}}{vector of integers. Maximum numbers of
events (tte endpoints) or patients (non-tte endpoints) at the final
analysis of each hypothesis when planning a trial. The actual numbers
could be different, which can be specified elsewhere.}

\item{\code{hypotheses}}{vector of characters. Names of hypotheses.}

\item{\code{silent}}{\code{TRUE} if muting all messages and not generating
plots.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-reset"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-reset}{}}}
\subsection{Method \code{reset()}}{
reset an object of class \code{GraphicalTesting} to original status
so that it can be reused.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$reset()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-is_valid_hid"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-is_valid_hid}{}}}
\subsection{Method \code{is_valid_hid()}}{
determine if index of a hypothesis is valid
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$is_valid_hid(hid)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid}}{an integer}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_hypothesis_name"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_hypothesis_name}{}}}
\subsection{Method \code{get_hypothesis_name()}}{
get name of a hypothesis given its index.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_hypothesis_name(hid)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid}}{an integer}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_weight"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_weight}{}}}
\subsection{Method \code{get_weight()}}{
return weight between two nodes
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_weight(hid1, hid2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid1}}{an integer}

\item{\code{hid2}}{an integer}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-set_weight"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-set_weight}{}}}
\subsection{Method \code{set_weight()}}{
update weight between two nodes
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$set_weight(hid1, hid2, value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid1}}{an integer}

\item{\code{hid2}}{an integer}

\item{\code{value}}{numeric value to be set as a weight two nodes}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_alpha"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_alpha}{}}}
\subsection{Method \code{get_alpha()}}{
return alpha allocated to a hypothesis when calling this function.
Note that a function can be called several time with the graph is
updated dynamically. Thus, returned alpha can be different even for
the same \code{hid}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_alpha(hid)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid}}{an integer}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-set_alpha"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-set_alpha}{}}}
\subsection{Method \code{set_alpha()}}{
update alpha of a hypothesis
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$set_alpha(hid, value)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid}}{integer. Index of a hypothesis}

\item{\code{value}}{numeric value to be allocated}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_hypotheses_ids"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_hypotheses_ids}{}}}
\subsection{Method \code{get_hypotheses_ids()}}{
return all valid \code{hid}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_hypotheses_ids()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_number_hypotheses"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_number_hypotheses}{}}}
\subsection{Method \code{get_number_hypotheses()}}{
return number of hypotheses, including those been rejected.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_number_hypotheses()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_hids_not_in_graph"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_hids_not_in_graph}{}}}
\subsection{Method \code{get_hids_not_in_graph()}}{
return index of hypotheses that are rejected.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_hids_not_in_graph()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_testable_hypotheses"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_testable_hypotheses}{}}}
\subsection{Method \code{get_testable_hypotheses()}}{
return index of hypotheses with non-zero alphas, thus can be tested
at the current stage.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_testable_hypotheses()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-has_testable_hypotheses"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-has_testable_hypotheses}{}}}
\subsection{Method \code{has_testable_hypotheses()}}{
determine whether at least one hypothesis is testable.
If return \code{FALSE}, the testing procedure is completed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$has_testable_hypotheses()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-is_in_graph"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-is_in_graph}{}}}
\subsection{Method \code{is_in_graph()}}{
determine whether a hypothesis is not yet rejected (in graph).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$is_in_graph(hid)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid}}{integer. Index of a hypothesis}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-is_testable"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-is_testable}{}}}
\subsection{Method \code{is_testable()}}{
determine whether a hypothesis has a non-zero alpha allocated.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$is_testable(hid)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hid}}{integer. Index of a hypothesis}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_hid"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_hid}{}}}
\subsection{Method \code{get_hid()}}{
convert hypothesis's name into (unique) index.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_hid(hypothesis)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hypothesis}}{character. Name of a hypothesis. It is different from
\code{hid}, which is an index.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-reject_a_hypothesis"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-reject_a_hypothesis}{}}}
\subsection{Method \code{reject_a_hypothesis()}}{
remove a node from graph when a hypothesis is rejected
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$reject_a_hypothesis(hypothesis)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{hypothesis}}{name of a hypothesis. It is different from
\code{hid}, which is an index.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-set_trajectory"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-set_trajectory}{}}}
\subsection{Method \code{set_trajectory()}}{
save new testing results at current stage
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$set_trajectory(result)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{result}}{a data frame of specific columns.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_trajectory"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_trajectory}{}}}
\subsection{Method \code{get_trajectory()}}{
return testing results by the time this function is called.
Note that graphical test is carried out in a sequential manner.
Users may want to review the results anytime. Value returned
by this function can possibly vary over time.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_trajectory()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-test_hypotheses"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-test_hypotheses}{}}}
\subsection{Method \code{test_hypotheses()}}{
test hypotheses using p-values (and other information in \code{stats})
base on the current graph. All rows should have the same order
number.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$test_hypotheses(stats)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{stats}}{a data frame. It must contain the following columns:
\describe{
\item{\code{order}}{integer. P-values (among others) of hypotheses that
can be tested at the same time (e.g., an interim, or final analysis)
should be labeled with the same order number.
If a hypothesis is not tested at a stage,
simply don't put it in \code{stats} with that order number.}
\item{\code{hypotheses}}{character. Name of hypotheses to be tested. They
should be identical to those when calling \code{GraphicalTesting$new}.}
\item{\code{p}}{nominal p-values.}
\item{\code{info}}{observed number of events or samples at test. These will
be used to compute information fractions in group sequential design.}
\item{\code{max_info}}{integers. Maximum information at test. At interim,
\code{max_info} should be equal to \code{planned_max_info} when
calling \code{GraphicalTesting$new}. At the final stage of a
hypothesis, one can update it with observed numbers.}
}}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-test"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-test}{}}}
\subsection{Method \code{test()}}{
test hypotheses using p-values (and other information in \code{stats})
base on the current graph. Users can call this function multiple times.
P-values of the same order should be passed through \code{stats}
together. P-values of multiple orders can be passed together as well.
For example, if users only have p-values at current stage, they can call
this function and update the graph accordingly. In this case, column
\code{order} in \code{stats} is a constant. They can call this
function again when p-values of next stage is available, where
\code{order} is another integer. In simulation, if p-values of all
stages are on hand, users can call this function to
test them all in a single pass. In this case, column \code{order} in
\code{stats} can have different values.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$test(stats)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{stats}}{a data frame. It must contain the following columns:
\describe{
\item{\code{order}}{integer. P-values (among others) of hypotheses that
can be tested at the same time (e.g., an interim, or final analysis)
should be labeled with the same order number.
If a hypothesis is not tested at a stage,
simply don't put it in \code{stats} with that order number.
If all p-values in \code{stats} are tested at the same stage, \code{order}
can be absent.}
\item{\code{hypotheses}}{character. Name of hypotheses to be tested. They
should be identical to those when calling
\code{GraphicalTesting$new}.}
\item{\code{p}}{nominal p-values.}
\item{\code{info}}{observed number of events or samples at test. These will
be used to compute information fractions in group sequential design.}
\item{\code{max_info}}{integers. Maximum information at test. At interim,
\code{max_info} should be equal to \code{planned_max_info} when
calling \code{GraphicalTesting$new}. At the final stage of a
hypothesis, one can update it with observed numbers.}
\item{\code{alpha_spent}}{accumulative proportion of allocated alpha
to be spent if \code{alpha_spending = "asUser"}. Set it to
\code{NA_real_} otherwise. If no hypothesis uses \code{"asUser"} in
\code{stats}, this column could be ignored. }
}}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
a data frame returned by \code{get_current_testing_results}.
It contains details of each of the testing steps.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_current_testing_results"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_current_testing_results}{}}}
\subsection{Method \code{get_current_testing_results()}}{
return testing results with details by the time this function
is called. This function can be called by users by multiple
times, thus the returned value varies over time.
This function is called by \code{GraphicalTesting::test},
and returns a data frame consisting of columns
\describe{
\item{\code{hypothesis}}{name of hypotheses.}
\item{\code{obs_p_value}}{observed p-values.}
\item{\code{max_allocated_alpha}}{maximum allocated alpha for the hypothesis.}
\item{\code{decision}}{\code{'reject'} or \code{'accept'} the hypotheses.}
\item{\code{stages}}{stage of a hypothesis. }
\item{\code{order}}{order number that this hypothesis is tested for the last time.
It is different from \code{stages}.}
\item{\code{typeOfDesign}}{name of alpha spending functions.}
}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_current_testing_results()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-get_current_decision"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-get_current_decision}{}}}
\subsection{Method \code{get_current_decision()}}{
get current decisions for all hypotheses. Returned value could
changes over time because it depends on the stages being tested already.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$get_current_decision()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
a named vector of values \code{"accept"} or \code{"reject"}.
Note that if a hypothesis is not yet tested when calling this function,
the decision for that hypothesis would be \code{"accept"}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-print"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-print}{}}}
\subsection{Method \code{print()}}{
generic function for \code{print}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$print(graph = TRUE, trajectory = TRUE, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{graph}}{logic. \code{TRUE} if visualizing the current graph,
which can vary over time.}

\item{\code{trajectory}}{logic. \code{TRUE} if print the current data frame of
trajectory, which can vary over time.}

\item{\code{...}}{other arguments supported in \code{gMCPLite::hGraph},
e.g., \code{trhw} and \code{trhh} to control the size of transition box,
and \code{trdigits} to control the digits displayed for transition
weights.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GraphicalTestingProcedure-clone"></a>}}
\if{latex}{\out{\hypertarget{method-GraphicalTestingProcedure-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GraphicalTesting$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
