% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blackmarbler.R
\name{bm_raster}
\alias{bm_raster}
\title{Make Black Marble Raster}
\usage{
bm_raster(
  roi_sf,
  product_id,
  date,
  bearer,
  variable = NULL,
  quality_flag_rm = NULL,
  check_all_tiles_exist = TRUE,
  interpol_na = FALSE,
  output_location_type = "memory",
  file_dir = NULL,
  file_prefix = NULL,
  file_skip_if_exists = TRUE,
  file_return_null = FALSE,
  h5_dir = NULL,
  download_method = "httr",
  quiet = FALSE,
  ...
)
}
\arguments{
\item{roi_sf}{Region of interest; sf polygon. Must be in the \href{https://epsg.io/4326}{WGS 84 (epsg:4326)} coordinate reference system.}

\item{product_id}{One of the following:
\itemize{
\item \code{"VNP46A1"}: Daily (raw)
\item \code{"VNP46A2"}: Daily (corrected)
\item \code{"VNP46A3"}: Monthly
\item \code{"VNP46A4"}: Annual
}}

\item{date}{Date of raster data. Entering one date will produce a \code{SpatRaster} object. Entering multiple dates will produce a \code{SpatRaster} object with multiple bands; one band per date.
\itemize{
\item For \code{product_id}s \code{"VNP46A1"} and \code{"VNP46A2"}, a date (eg, \code{"2021-10-03"}).
\item For \code{product_id} \code{"VNP46A3"}, a date or year-month (e.g., \code{"2021-10-01"}, where the day will be ignored, or \code{"2021-10"}).
\item For \code{product_id} \code{"VNP46A4"}, year or date  (e.g., \code{"2021-10-01"}, where the month and day will be ignored, or \code{2021}).
}}

\item{bearer}{NASA bearer token. For instructions on how to create a token, see \href{https://github.com/worldbank/blackmarbler#bearer-token-}{here}.}

\item{variable}{Variable to used to create raster (default: \code{NULL}). If \code{NULL}, uses the following default variables:
\itemize{
\item For \code{product_id} \verb{:VNP46A1"}, uses \code{DNB_At_Sensor_Radiance_500m}.
\item For \code{product_id} \code{"VNP46A2"}, uses \code{Gap_Filled_DNB_BRDF-Corrected_NTL}.
\item For \code{product_id}s \code{"VNP46A3"} and \code{"VNP46A4"}, uses \code{NearNadir_Composite_Snow_Free}.
To see all variable choices, set \code{variable = ""} (this will create an error message that lists all valid variables). For additional information on variable choices, see \href{https://ladsweb.modaps.eosdis.nasa.gov/api/v2/content/archives/Document\%20Archive/Science\%20Data\%20Product\%20Documentation/VIIRS_Black_Marble_UG_v1.2_April_2021.pdf}{here}; for \code{VNP46A1}, see Table 3; for \code{VNP46A2} see Table 6; for \code{VNP46A3} and \code{VNP46A4}, see Table 9.
}}

\item{quality_flag_rm}{Quality flag values to use to set values to \code{NA}. Each pixel has a quality flag value, where low quality values can be removed. Values are set to \code{NA} for each value in the \code{quality_flag_rm} vector. Note that \code{quality_flag_rm} does not apply for \code{VNP46A1}. (Default: \code{NULL}).

For \code{VNP46A2} (daily data):
\itemize{
\item \code{0}: High-quality
\item \code{1}: Poor-quality - Main Algorithm (Outlier, Potential cloud contamination or other issues)
\item \code{2}: Poor-quality - Main Algorithm (high solar zenith angle 102-108 degrees)
\item \code{3}: Poor-quality - Main Algorithm (Lunar eclipse)
\item \code{4}: Poor-quality - Main Algorithm (Aurora)
\item \code{5}: Poor-quality - Main Algorithm (Glint)
}

For \code{VNP46A3} and \code{VNP46A4} (monthly and annual data):
\itemize{
\item \code{0}: Good-quality, The number of observations used for the composite is larger than 3
\item \code{1}: Poor-quality, The number of observations used for the composite is less than or equal to 3
\item \code{2}: Gap filled NTL based on historical data
}}

\item{check_all_tiles_exist}{Check whether all Black Marble nighttime light tiles exist for the region of interest. Sometimes not all tiles are available, so the full region of interest may not be covered. If \code{TRUE}, skips cases where not all tiles are available. (Default: \code{TRUE}).}

\item{interpol_na}{When data for more than one date is downloaded, whether to interpolate \code{NA} values using the \code{terra::approximate} function. Additional arguments for the \code{terra::approximate} function can also be passed into \code{bm_raster} (eg, \code{method}, \code{rule}, \code{f}, \code{ties}, \code{z}, \code{NA_rule}). (Default: \code{FALSE}).}

\item{output_location_type}{Where to produce output; either \code{memory} or \code{file}. If \code{memory}, functions returns a raster in R. If \code{file}, function exports a \code{.tif} file and returns \code{NULL}.
For \code{output_location_type = file}:}

\item{file_dir}{The directory where data should be exported (default: \code{NULL}, so the working directory will be used)}

\item{file_prefix}{Prefix to add to the file to be saved. The file will be saved as the following: \verb{[file_prefix][product_id]_t[date].tif}}

\item{file_skip_if_exists}{Whether the function should first check wither the file already exists, and to skip downloading or extracting data if the data for that date if the file already exists (default: \code{TRUE}).}

\item{file_return_null}{Whether to return \code{NULL} instead of a \code{SpatRaster}. When \code{output_location_type = 'file'}, the function will export data to the \code{file_dir} directory. When \code{file_return_null = FALSE}, the function will also return a \code{SpatRaster} of the queried data---so the data is available in R memory. Setting \code{file_return_null = TRUE}, data will be saved to \code{file_dir} but no data will be returned by the function to R memory (default: \code{FALSE}).}

\item{h5_dir}{Black Marble data are originally downloaded as \code{h5} files. If \code{h5_dir = NULL}, the function downloads to a temporary directory then deletes the directory. If \code{h5_dir} is set to a path, \code{h5} files are saved to that directory and not deleted. The function will then check if the needed \code{h5} file already exists in the directory; if it exists, the function will not re-download the \code{h5} file.}

\item{download_method}{Method to download data from NASA LAADS Archive: "\code{httr}" or "\code{wget}". If \code{httr}, uses the \code{httr2} R package to download data. If \code{wget}, uses the \code{wget} command line tool. \code{httr} is fully integrated in R, while \code{wget} requires the \code{wget} system command. \code{wget} can be more efficient and can help avoid network issues. (Default: \code{"httr"}).}

\item{quiet}{Suppress output that show downloading progress and other messages. (Default: \code{FALSE}).}

\item{...}{Additional arguments for \code{terra::approximate}, if \code{interpol_na = TRUE}}
}
\value{
Raster
}
\description{
Make a raster of nighttime lights from \href{https://blackmarble.gsfc.nasa.gov/}{NASA Black Marble data}
}
\examples{
\dontrun{
# Define bearer token
bearer <- "BEARER-TOKEN-HERE"

# sf polygon of Ghana
library(geodata)
roi_sf <- gadm(country = "GHA", level=0, path = tempdir()) \%>\% st_as_sf()

# Daily data: raster for October 3, 2021
ken_20210205_r <- bm_raster(roi_sf = roi_sf,
                            product_id = "VNP46A2",
                            date = "2021-10-03",
                            bearer = bearer)

# Monthly data: raster for March 2021
ken_202103_r <- bm_raster(roi_sf = roi_sf,
                          product_id = "VNP46A3",
                          date = "2021-03-01",
                          bearer = bearer)

# Annual data: raster for 2021
ken_2021_r <- bm_raster(roi_sf = roi_sf,
                        product_id = "VNP46A4",
                        date = 2021,
                        bearer = bearer)
}

}
\author{
Robert Marty \href{mailto:rmarty@worldbank.org}{rmarty@worldbank.org}
}
