% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/admix_prop_1d_circular.R
\name{admix_prop_1d_circular}
\alias{admix_prop_1d_circular}
\title{Construct admixture proportion matrix for circular 1D geography}
\usage{
admix_prop_1d_circular(
  n_ind,
  k_subpops,
  sigma = NA,
  coord_ind_first = 2 * pi/(2 * n_ind),
  coord_ind_last = 2 * pi * (1 - 1/(2 * n_ind)),
  bias_coeff = NA,
  coanc_subpops = NULL,
  fst = NA
)
}
\arguments{
\item{n_ind}{Number of individuals}

\item{k_subpops}{Number of intermediate subpopulations}

\item{sigma}{Spread of intermediate subpopulations (approximate standard deviation of Von Mises densities, see above)
The edge cases \code{sigma = 0} and \code{sigma = Inf} are handled appropriately!}

\item{coord_ind_first}{Location of first individual}

\item{coord_ind_last}{Location of last individual

OPTIONS FOR BIAS COEFFICIENT VERSION}

\item{bias_coeff}{If \code{sigma} is \code{NA}, this bias coefficient is required.}

\item{coanc_subpops}{If \code{sigma} is \code{NA}, this intermediate subpops coancestry is required.
It can be provided as a \code{k_subpops}-by-\code{k_subpops} matrix, a length-\code{k_subpops} population inbreeding vector (for independent subpopulations, where between-subpop coancestries are zero) or scalar (if population inbreeding values are all equal and coancestries are zero).
This \code{coanc_subpops} can be in the wrong scale (it cancels out in calculations), which is returned corrected, to result in the desired \code{fst} (next).}

\item{fst}{If \code{sigma} is \code{NA}, this FST of the admixed individuals is required.}
}
\value{
If \code{sigma} was provided, returns the \code{n_ind}-by-\code{k_subpops} admixture proportion matrix (\code{admix_proportions}).
If \code{sigma} is missing, returns a named list containing:
\itemize{
\item \code{admix_proportions}: the \code{n_ind}-by-\code{k_subpops} admixture proportion matrix.
If \code{coanc_subpops} had names, they are copied to the columns of this matrix.
\item \code{coanc_subpops}: the input \code{coanc_subpops} rescaled.
\item \code{sigma}: the fit value of the spread of intermediate subpopulations
\item \code{coanc_factor}: multiplicative factor used to rescale \code{coanc_subpops}
}
}
\description{
Assumes \code{k_subpops} intermediate subpopulations placed along a circumference (the [\code{0}, \code{2 * pi}] line that wraps around) with even spacing spread by random walks (see details below), then \code{n_ind} individuals sampled equally spaced in [\code{coord_ind_first},\code{coord_ind_last}] (default [\code{0}, \code{2 * pi}] with a small gap so first and last individual do not overlap) draw their admixture proportions relative to the Von Mises density that models the random walks of each of these intermediate subpopulations.
The spread of the random walks is \code{sigma = 1 / sqrt(kappa)} of the Von Mises density.
If \code{sigma} is missing, it can be set indirectly by providing three variables: (1) the desired bias coefficient \code{bias_coeff}, (2) the coancestry matrix of the intermediate subpopulations \code{coanc_subpops} (up to a scalar factor), and (3) the final \code{fst} of the admixed individuals (see details below).
}
\details{
Assuming the full range of [\code{0}, \code{2 * pi}] is considered, and the first and last individuals do not overlap, the gap between individuals is \code{delta = 2 * pi / n}.
To not have any individuals on the edge, we place the first individual at \code{delta / 2} and the last at \code{2 * pi - delta / 2}.
The location of subpopulation \code{j} is \code{delta / 2 + ( j - 1/2 ) / k * (2 * pi - delta)}, chosen to agree with the default correspondence between individuals and subpopulations of the linear 1D geography admixture scenario (\code{\link[=admix_prop_1d_linear]{admix_prop_1d_linear()}}).

If \code{sigma} is \code{NA}, its value is determined from the desired \code{bias_coeff}, \code{coanc_subpops} up to a scalar factor, and \code{fst}.
Uniform weights for the final generalized FST are assumed.
The scale of \code{coanc_subpops} is irrelevant because it cancels out in \code{bias_coeff}; after \code{sigma} is found, \code{coanc_subpops} is rescaled to give the desired final FST.
However, the function stops if any rescaled \code{coanc_subpops} values are greater than 1, which are not allowed since they are IBD probabilities.
}
\examples{
# admixture matrix for 1000 individuals drawing alleles from 10 subpops
# simple version: spread of about 2 standard deviations along the circular 1D geography
# (just set sigma)
admix_proportions <- admix_prop_1d_circular(n_ind = 1000, k_subpops = 10, sigma = 2)

# advanced version: a similar model but with a bias coefficient of exactly 1/2
# (must provide bias_coeff, coanc_subpops, and fst in lieu of sigma)
k_subpops <- 10
# FST vector for intermediate independent subpops, up to a factor (will be rescaled below)
coanc_subpops <- 1 : k_subpops
obj <- admix_prop_1d_circular(
    n_ind = 1000,
    k_subpops = k_subpops,
    bias_coeff = 0.5,
    coanc_subpops = coanc_subpops,
    fst = 0.1 # desired final FST of admixed individuals
)

# in this case return value is a named list with three items:
admix_proportions <- obj$admix_proportions

# rescaled coancestry data (matrix or vector) for intermediate subpops
coanc_subpops <- obj$coanc_subpops

# and the sigma that gives the desired bias_coeff and final FST
sigma <- obj$sigma

}
