##
# ------------------------------------------------------------------------
#
# "aidedboot(x,XI,g,B,order,kernel,bandwidth)" --
#
# Aided Frequency Bootstrap (AFB).
#
# ------------------------------------------------------------------------
##
#' @aliases aidedboot
#' @title Aided Frequency Bootstrap
#' @description The Aided Frequency Bootstrap (AFB) is a variation of the Frequency
#' Domain Bootstrap (FDB). The idea is to fit a sieve AR(p) model and to generate
#' the corresponding bootstrapped time series (by resampling centered residuals) with 
#' periodogram \eqn{I^{\ast}_{AR}}.
#' Then the we estimate the quotient of the two 
#' spectral densities \eqn{q(\omega)=\frac{f(\omega)}{f_{AR}(\omega)}} and generate 
#' bootstrap periodogram by multiplying \eqn{I^{\ast}_{AR}} by this quantity 
#' \eqn{q(\omega)=\frac{f(\omega)}{f_{AR}(\omega)}}.
#' 
#' @param x A numeric vector representing a time series.
#' @param XI A list of functions defined on the interval \eqn{[0, \pi]}.
#' @param g A numeric function taking \code{length(XI)} as arguments.
#' @param B A positive integer; the number of bootstrap samples.
#' @param order The order of the autoregressive sieve process (integer). 
#'        If not specified, it is set by default as
#'        \deqn{\left \lfloor  4 * (length(x) \log(length(x)))^{1/4}\right \rfloor.}
#' @param kernel A character string specifying the smoothing kernel. The possible choices are: 
#' \code{"normal"}, \code{"box"} and \code{"epanechnikov"}.
#' @param bandwidth The kernel bandwidth smoothing parameter. If missing, the bandwidth 
#' is automaticly computed by \code{\link{bandw1}}.
#' @details 
#' The idea underlying the Aided Frequency Bootstrap is
#' importance sampling. It was introduced by Kreiss and Paparoditis (2003) and
#' allows to better mimic the asymptotic covariance structure of the
#' periodogram in the bootstrap world. Kreiss and Paparoditis (2003)
#' considered a spectral density which is easy to estimate (typically based on
#' a sieve AR representation of the time series), say \eqn{f_{AR}(\omega)}.
#' The argument \code{x} is supposed to be a sample of a real valued zero-mean
#' stationary time series.
#'
#' The autoregressive sieve process of order \eqn{l=l(n)} is modelled as
#' \deqn{X_{t}=\sum_{k=1}^{l}\psi_{k}X_{t-k}+\epsilon_{t}}
#' with \eqn{E(\epsilon_{t})=0, Var(\epsilon_{t})=\sigma^{2}(l)}.
#'
#' We estimate functionals of the spectral density \eqn{T(f)} of the form
#' \deqn{
#'  T(f)=g(A(\xi,f))
#' }
#' where \eqn{g} is a third order differentiable function,
#' \deqn{A(\xi,f)=\left(  \int_{0}^{\pi}\xi_{1}(\omega)f(\omega)d\omega,\int_{0}^{\pi
#' }\xi_{2}(\omega)f(\omega)d\omega,\dots,\int_{0}^{\pi}\xi_{p}(\omega
#' )f(\omega)d\omega\right)
#' }
#' and
#' \deqn{
#'  \xi=(\xi_{1},\dots,\xi_{p}): [0,\pi] \rightarrow R^p.
#' } 
#'
#' If the \code{order} argument is not specified, its default value is
#' \eqn{l=\left \lfloor  (4*(n\log(n))^{1/4})\right \rfloor}, where \code{n} is the length of \code{x}.
#'
#' The \code{kernel} argument has the same meaning as in the \code{\link{freqboot}}
#' function.
#' @return \code{aidedboot} returns an object of class \code{boodd} 
#' (see \link{class.boodd}).
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek)- submitted.
#'
#' Kreiss, J.-P. and Paparoditis, E. (2003). Autoregressive aided
#' periodogram bootstrap for time series. \emph{Ann. Stat.} \bold{31} 1923--1955.
#' 
#' @seealso \code{\link{freqboot}}.
#' @keywords Bootstrap "Frequency domain" Periodogram
#' @export
#' @examples 
#' n <- 200
#' x <- arima.sim(list(order=c(4,0,0),ar=c(0.7,0.4,-0.3,-0.1)),n=n)
#' B <- 299
#' one <- function(x) {1}
#' XI <- list(cos,one)
#' g <- function(x,y) {return(x/y)}
#' ord <- 2*floor(n^(1/3))
#' boo <- aidedboot(x,XI,g,B,order=ord) 
#' plot(boo)
##
aidedboot <- function(x,XI,g,B,order=NULL,kernel="normal",bandwidth) {
  n <- length(x)
  p <- length(XI)
  # Check the arguments
  for (i in 1:p) {
    if (!is.list(XI) | !is.function(XI[[i]])) {
      stop("XI must be a list of functions")
    }
  }
  if (length(formalArgs(g)) != p) {
    stop("g must be a numeric function with as many arguments as the length of XI (",p,")")
  }
  y <- do.call(g,as.list(rep(1,p)))
  if (!is.vector(y)) {
    stop("Function 'g' must return a vector")
  }
  if (missing(bandwidth)) {h <- bandw1(x)}
  if (is.null(order)) {
    #ln <- floor(n^0.25/sqrt(log(n)))
    ln <- floor(4*(n/log(n))^0.25)
  } else {
    ln <- order
  }
  if (n <= ln) {
    stop("length of series must be greater than order")
  }
  
  # Initial periodograms
  x <- ts(x,frequency=1)
  P <- spec.pgram(x,plot=FALSE,taper=0,fast=FALSE,detrend=FALSE)
  specs <- P$spec/(2*pi)
  freqs <- P$freq*2*pi
  n0 <- length(specs)
  P0 <- sum(x)^2/(2*pi*n)
  I_n <- c(rev(specs),P0,specs)
  ljn <- c(-rev(freqs),0,freqs)
  
  # Precompute the \xi_i(\lambda_{jn})
  xiljn <- matrix(nrow=p,ncol=n0)
  for (i in 1:p) {
    xiljn[i,] <- XI[[i]](freqs)
  }
  
  # Precompute the smoothing coefficients
  smc <- smoothingCoefficients(n,h,kernel)
  
  # Initial statistic
  V <- numeric(p)
  for (i in 1:p) {
    V[i] <- mean(xiljn[i,]*specs)
  }
  y <- do.call(g,as.list(V))
  len <- length(y)
  res <- matrix(nrow=B,ncol=len)
  cnames <- names(y)
  
  # Fit the linear autoregressive sieve process
  fit <- arima(x, order=c(ln,0,0))
  coeffs <- coef(fit)
  psi <- coeffs[1:ln]
  eps_hat <- residuals(fit)
  # Remove the first ln values and center on the mean
  eps_hat <- eps_hat[-(1:ln)]
  eps_tilde <- eps_hat - sum(eps_hat)/(n-ln)
  
  # Bootstrap
  for (j in 1:B) {
    # Draw epsilon values randomly with replacement
    eps_star <- sample(eps_tilde,n-ln,replace=TRUE)
    
    # Reconstruct the bootstrapped series
    xstar <- numeric(n)
    xstar[1:ln] <- x[1:ln]
    for (i in (ln+1):n) {
      xstar[i] <- sum(psi*xstar[(i-1):(i-ln)]) + eps_star[i-ln]
    }
    
    # Compute the periodograms of the xstar series
    P <- spec.pgram(ts(xstar,frequency=1),plot=FALSE,taper=0,fast=FALSE,detrend=FALSE)
    I_star_tilde <- P$spec/(2*pi)
    
    # Compute \hat{f} for the Fourier frequencies
    sigma2 <- sum(eps_star^2)/(n-ln)
    fhat <- numeric(n0)
    J <- 1:ln
    for (i in 1:n0) {
      f <- freqs[i]
      eijf <- complex(modulus=1,argument=-f*J)
      fhat[i] <- Mod(1-sum(psi*eijf))^(-2)*sigma2/(2*pi)
    }
    # Complete fhat by symmetry
    fhat0 <- Mod(1-sum(psi))^(-2)*sigma2/(2*pi)
    fhat <- c(rev(fhat),fhat0,fhat)
    
    # Compute \hat{q} for the Fourier frequencies
    QIf <- I_n/fhat
    qhat <- numeric(n0)
    for (i in 1:n0) {
      K <- smc[(n0-i+1):(3*n0-i+1)]
      qhat[i] <- 2*pi*mean(K*QIf)/h
    }
    I_star <- I_star_tilde*qhat
    V <- numeric(p)
    for (i in 1:p) {
      V[i] <- mean(xiljn[i,]*I_star)
    }
    res[j,] <- do.call(g,as.list(V))
  }
  
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- "aided"
  attr(obj,"func") <- g
  return(obj)
}