##
# ------------------------------------------------------------------------
#
# "blockboot(x,func,B,length.block,method,period,...)" --
#
# Block bootstrap for time series. Method names maybe abbreviated.
#
# ------------------------------------------------------------------------
##
#' @aliases blockboot
#' @title Block Bootstrap
#' @description The function applies block bootstrap methods to a time series.
#' 
#' This function allows the following block bootstrap methods to be
#' used: the Moving Block Bootstrap (Kunsch (1989), Liu and Singh (1992)),
#' the Nonoverlapping Block Bootstrap (Carlstein (1986)), the Circular
#' Block Bootstrap (Politis and Romano (1992)), and the Stationary Bootstrap
#' (Politis and Romano (1994)).
#' @param x A vector or a time series.
#' @param func The function to apply to each sample.
#' @param B A positive integer; the number of bootstrap replications.
#' @param length.block A positive integer; the length of the blocks.
#' @param ... Optional additional arguments for the \code{func} function.
#' @param method The block bootstrap method. The possible values of the \code{method} argument are: 
#'   \code{"movingblock"}, \code{"nonoverlapping"},
#'  \code{"circular"} or \code{"stationary"}.
#'  If it is not specified, the default method is \emph{"movingblock"}.
#'  Method names may be abbreviated.
#' @param moon Integer or \code{NULL}. When \code{moon = NULL} (default), \code{blockboot}
#'  performs a regular block bootstrap without subsampling. 
#'             If \code{moon} is equal to some integer value, 
#'             the function creates block bootstrap samples of size \code{moon}, drawing blocks of 
#'             length \code{length.block}. 
#'             Ensure that \code{moon} is less than 
#'             \code{n - 5}, where \code{n} is the size of the data.
#' @param replace Logical. If `replace = TRUE` (default), the function performs block 
#' bootstrap with replacement. If `replace = FALSE`, it performs Block Bootstrap 
#' Subsampling without replacement. In this case \code{moon} should be specified.
#' @details Nonoverlapping Block Bootstrap (NBB) consists in cutting the original time
#' series into nonoverlapping blocks of fixed length \code{length.block} and in 
#' resampling these blocks to reconstruct a 
#' bootstrap time series.
#' Moving Block Bootstrap (MBB) consists in drawing independently overlapping blocks 
#' of fixed size \code{length.block} to reconstruct a bootstrap time series of 
#' length of the original process.
#' Circular Block Bootstrap (CBB) consists in wrapping the data on a circle and to 
#' create the corresponding overlapping blocks, so that each value of the time series 
#' appears globally the same number of times in all the blocks. This generally reduce the 
#' bias of the bootstrap distribution. 
#' Stationary Bootstrap (SB) is based on blocks with random length, which ensure that 
#' the bootstrap sample is stationary.
#' @return An object of class \code{boodd}. 
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Carlstein E. (1986). The use of subseries methods for
#' estimating the variance of a general statistic from a stationary time series.
#' \emph{Annals of Statist.}, \bold{14}, 1171-1179.
#' 
#' Künsch, H. (1989). The jackknife and the bootstrap for
#' general stationary observations. \emph{Ann. Statist.}, 17, 1217-1241.
#' 
#' Liu, R. and Singh, K. (1992). Moving block jackknife and
#' bootstrap capture weak dependence. \emph{Exploring the Limits of Bootstrap.},
#' Series in Probab. Math. Statist. Wiley, New York, pp 225-248.
#' 
#' Politis, D.N. and Romano, J.P. (1994). The stationary
#' bootstrap. \emph{J. Amer. Statist. Assoc.}, \bold{89}, 1303--1313.
#' 
#' Politis, D.N. and Romano, J.P. (1992).
#' A circular block-resampling procedure for stationary data.
#' \emph{Exploring the Limits of Bootstrap.},
#' Series in Probab. Math. Statist. Wiley, New York, pp 263-270.
#' 
#' 
#' 
#' @seealso {\code{\link{boots}},
#' \code{\link{bootsemi}},
#' \code{\link{plot.boodd}},
#' \code{\link{confint.boodd}},
#' \code{\link{fieldboot}},
#' \code{\link{jackVarBlock}}}.
#' @keywords "Bootstrap" "Moving block"
#' @export
#' @examples 
#' B <- 999
#' data(airquality)
#' x <- airquality$Wind
#' n <- length(x)
#' b <- floor(sqrt(n))
#' boo1 <- blockboot(x,mean,B,b,method="moving")
#' plot(boo1,main="MBB", nclass=30)
#' confint(boo1, method="all")
##

blockboot<-function (x, func, B, length.block = NULL, method = c("movingblock", 
    "nonoverlapping", "circular", "stationary"), moon=NULL, replace="TRUE", ...) 
{
  method <- match.arg(method)
  if (is.null(length.block)) {
    defaultlen <- length(x)^(1/3)
    if (method[1] == "circular") {
      blstar <- b.star(x)[2]
    }
    else {
      blstar <- b.star(x)[1]
    }
    length.block <- ceiling(max(c(blstar, defaultlen)))
  }
  
  if (length.block >= length(x)) {
    stop("the block length must be less than the size of the time series")
  }
  y <- func(x, ...)
  if (!is.vector(y)) {
    stop("Function 'func' must return a vector")
  }
  len <- length(y)
  cnames <- names(y)
  
  
  if (method[1] == "movingblock") {
    res <- blockboot.mb(x, func, B, length.block, moon, replace, ...)
  }
  else if (method[1] == "nonoverlapping") {
    res <- blockboot.nooverlap(x, func, B, length.block, moon, replace, ...)
  }
  else if (method[1] == "circular") {
    res <- blockboot.circular(x, func, B, length.block, moon, replace, ...)
  }
  else if (method[1] == "stationary") {
    res <- blockboot.stationary(x, func, B, length.block, moon, replace, ...)
  }
  if (len == 1) {
    res <- as.vector(res)
  }
  else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  obj <- list(s = res, Tn = y)
  class(obj) <- "boodd"
  if (is.null(moon))  attr(obj, "kind") <- c("block", method[1])
  else attr(obj, "kind") <- c("sub-block", method[1])  
  attr(obj, "func") <- func
  return(obj)
}


##
# ------------------------------------------------------------------------
#
# "blockboot.mb <- function(x,func,B,length.block,...)" --
#
# Block bootstrap for time series using moving blocks.
#
# ------------------------------------------------------------------------
##

blockboot.mb<-function (x, func, B, length.block, moonsize, rep, ...) 
{
  n <- length(x)
  y <- func(x, ...)
  b <- length.block
  len <- length(y)
  res <- matrix(nrow = B, ncol = len)
  q <- n - b + 1
  if (is.null(moonsize)) s <- ((n - 1)%/%b) + 1 
  else {
    if (moonsize>n-5) stop("moonsize or subsampling size should be much smaller than n")
    else s <-((moonsize - 1)%/%b) + 1
  }
  nx <- numeric(s * b)
  for (i in 1:B) {
    bb <- sample(1:q, size = s, rep)
    pos <- 1
    for (idx in bb) {
      nx[pos:(pos + b - 1)] <- x[idx:(idx + b - 1)]
      pos <- pos + b
    }
    if (is.null(moonsize)) res[i, ] <- func(nx[1:n], ...) 
    else  res[i, ] <- func(nx[1:moonsize], ...)
  }
  return(res)
}

##
# ------------------------------------------------------------------------
#
# "blockboot.nooverlap <- function(x,func,B,length.block,...)" --
#
# Block bootstrap for time series using non overlapping blocks.
#
# ------------------------------------------------------------------------
##

blockboot.nooverlap<-function (x, func, B, length.block, moonsize, rep, ...) 
{
  n <- length(x)
  y <- func(x, ...)
  b <- length.block
  len <- length(y)
  res <- matrix(nrow = B, ncol = len)
  q <- n%/%b
  q0<-q
  
  if (is.null(moonsize)) s <- ((n - 1)%/%b) + 1 
  else {
    if (moonsize>n-5) stop("moonsize or subsampling size should be much smaller than n")
    else { 
      s <-((moonsize - 1)%/%b) + 1
      q <-moonsize%/%b
    }
  }
  nx <- numeric(s * b)
  for (i in 1:B) {
    bb <- sample(1:q0, size = q + 1, rep)
    pos <- 1
    for (idx in bb) {
      start <- (idx - 1) * b + 1
      nx[pos:(pos + b - 1)] <- x[start:(start + b - 1)]
      pos <- pos + b
    }
    if (is.null(moonsize)) res[i, ] <- func(nx[1:n], ...) 
    else  res[i, ] <- func(nx[1:moonsize], ...)
  }
  return(res)
}


##
# ------------------------------------------------------------------------
#
# "blockboot.circular <- function(x,func,B,length.block,...)" --
#
# Block bootstrap for time series using circular blocks.
#
# ------------------------------------------------------------------------
##

blockboot.circular<-function (x, func, B, length.block,moonsize, rep, ...)
{
  n <- length(x)
  y <- func(x, ...)
  b <- length.block
  len <- length(y)
  res <- matrix(nrow = B, ncol = len)
  x <- c(x, x[1:(b - 1)])
  if (is.null(moonsize)) s <- ((n - 1)%/%b) + 1 
  else {
    if (moonsize>n-5) stop("moonsize or subsampling size should be much smaller than n")
    else s <-((moonsize - 1)%/%b) + 1
  }
  nx <- numeric(s * b)
  for (i in 1:B) {
    bb <- sample(1:n, size = s, rep)
    pos <- 1
    for (idx in bb) {
      nx[pos:(pos + b - 1)] <- x[idx:(idx + b - 1)]
      pos <- pos + b
    }
    if (is.null(moonsize)) res[i, ] <- func(nx[1:n], ...) 
    else  res[i, ] <- func(nx[1:moonsize], ...)
  }
  return(res)
}

##
# ------------------------------------------------------------------------
#
# "blockboot.stationary <- function(x,func,B,length.block,...)" --
#
# Block bootstrap for time series using random length blocks taken from
# random positions. Lengths are drawn from a geometric distribution with
# mean equal to 'length.block' (p=1/b) and positions are drawn uniformly.
#
# ------------------------------------------------------------------------
##

blockboot.stationary<-function (x, func, B, length.block, moonsize, rep, ...) 
{
  n <- length(x)
  y <- func(x, ...)
  b <- length.block
  p <- 1/b
  len <- length(y)
  res <- matrix(nrow = B, ncol = len)
  x <- c(x, x)
  for (i in 1:B) {
    L <- numeric(0)
    tot <- 0
    while (tot < n) {
      nl <- rgeom(1, p)
      if (nl > 0) {
        L <- c(L, nl)
        tot <- tot + nl
      }
    }
    K <- length(L)
    S <- sample(1:n, K, rep)
    nx <- numeric(tot)
    pos <- 1
    for (j in 1:K) {
      start <- S[j]
      b <- L[j]
      nx[pos:(pos + b - 1)] <- x[start:(start + b - 1)]
      pos <- pos + b
    }
    if (is.null(moonsize)) res[i, ] <- func(nx[1:n], ...) 
    else  res[i, ] <- func(nx[1:moonsize], ...)
  }
  return(res)
}

