% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstrap_functions.R
\name{ffmatrixmult}
\alias{ffmatrixmult}
\title{Matrix multiplication with "ff_matrix" or "matrix" inputs}
\usage{
ffmatrixmult(
  x,
  y = NULL,
  xt = FALSE,
  yt = FALSE,
  ram.output = FALSE,
  override.big.error = FALSE,
  ...
)
}
\arguments{
\item{x}{a matrix or ff_matrix}

\item{y}{a matrix or ff_matrix. If NULL, this is set equal to x, although a second copy of the matrix x is not actually stored.}

\item{xt}{should the x matrix be transposed before multiplying}

\item{yt}{should the y matrix be transposed before multiplying (e.g. \code{xt=TRUE}, \code{yt=FALSE} leads to \code{crossprod(x,y)}).}

\item{ram.output}{force output to be a normal matrix, as opposed to an object with class \code{ff}.}

\item{override.big.error}{If the dimension of the final output matrix is especially large, \code{ffmatrixmult} will abort, giving an error. This is meant to avoid the accidental creation of very large matrices. Set override.big.error=TRUE to bypass this error.}

\item{...}{passed to \link[ff]{ff}.}
}
\value{
A standard matrix, or a matrix with class \code{ff} if one of the input matrices has class \code{ff}.
}
\description{
A function for \code{crossprod(x,y)}, for \code{tcrossprod(x,y)}, or for regular matrix multiplication, that is compatible with \code{ff} matrices. Multiplication is done without creating new matrices for the transposes of \code{x} or \code{y}. Note, the crossproduct function can't be applied directly to objects with class \code{ff}.
}
\examples{
\dontrun{
 library(ff)
	#Tall data
	y_tall<-matrix(rnorm(5000),500,10) #y tall
	x_tall<-matrix(rnorm(5000),500,10)
	y_wide<-t(y_tall)
	x_wide<-t(x_tall)
	y_tall_ff<-as.ff(y_tall) #y tall and ff
	x_tall_ff<-as.ff(x_tall) 
	y_wide_ff<-as.ff(y_wide) #y tall and ff
	x_wide_ff<-as.ff(x_wide) 

 #Set options to ensure that block matrix algebra is actually done,
 #and the entire algebra isn't just one in one step.
 #Compare ffmatrixmult against output from standard methods
 options('ffbytesize'=100)

 #small final matrices
	#x'x
	range(  crossprod(x_tall) - ffmatrixmult(x_tall_ff, xt=TRUE)  )
	range(  tcrossprod(x_wide) - ffmatrixmult(x_wide_ff, yt=TRUE)  )
	range(  crossprod(x_tall,y_tall) - ffmatrixmult(x_tall_ff,y_tall_ff, xt=TRUE)  )
	range(  tcrossprod(x_wide,y_wide) - ffmatrixmult(x_wide_ff,y_wide_ff, yt=TRUE)  )
	range(  (x_wide\%*\%y_tall) - ffmatrixmult(x_wide_ff,y_tall_ff)  )

	#ff + small data
	s_tall <- matrix(rnorm(80),10,8) 
	s_wide <- matrix(rnorm(80),8,10) 

	#tall output
	range(  crossprod(x_wide, s_tall) - ffmatrixmult(x_wide_ff, s_tall,xt=TRUE)[]  )
	range(  tcrossprod(x_tall, s_wide) - ffmatrixmult(x_tall_ff, s_wide,yt=TRUE)[]  )
	range( x_tall\%*\%s_tall - ffmatrixmult(x_tall_ff, s_tall)[])

	#Wide output
	range(  crossprod(s_tall, y_wide) - ffmatrixmult( s_tall, y_wide_ff,xt=TRUE)[]  )
	range(  tcrossprod(s_wide, y_tall) - ffmatrixmult( s_wide,y_tall_ff,yt=TRUE)[]  )
	range( s_wide\%*\%y_wide - ffmatrixmult(s_wide,y_wide_ff)[])

 #Reset options for more practical use
 options('ffbytesize'=16777216)

}
}
