% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bvar.R, R/plot.bvar.R, R/predict.bvar.R
\name{bvar}
\alias{bvar}
\alias{plot.bvar}
\alias{predict.bvar}
\title{Bayesian Vector Autoregression Objects}
\usage{
bvar(
  data = NULL,
  exogen = NULL,
  y,
  x = NULL,
  A0 = NULL,
  A = NULL,
  B = NULL,
  C = NULL,
  Sigma = NULL
)

\method{plot}{bvar}(x, ci = 0.95, type = "hist", ...)

\method{predict}{bvar}(object, ..., n.ahead = 10, new_x = NULL, new_d = NULL, ci = 0.95)
}
\arguments{
\item{data}{the original time-series object of endogenous variables.}

\item{exogen}{the original time-series object of unmodelled variables.}

\item{y}{a time-series object of endogenous variables with \eqn{T} observations,
usually, a result of a call to \code{\link{gen_var}}.}

\item{x}{an object of class \code{"bvar"}, usually, a result of a call to \code{\link{draw_posterior}}.}

\item{A0}{either a \eqn{K^2 \times S} matrix of MCMC coefficient draws of structural parameters or
a named list, where element \code{coeffs} contains a \eqn{K^2 \times S} matrix of MCMC coefficient
draws of structural parameters and element \code{lambda} contains the corresponding draws of inclusion
parameters in case variable selection algorithms were employed. For time varying parameter models
the coefficient matrix must be \eqn{TK^2 \times S}. Draws of the error covariance matrix of the state
equation can be provided as a \eqn{K^2 \times S} matrix in an additional list element.}

\item{A}{either a \eqn{pK^2 \times S} matrix of MCMC coefficient draws of lagged endogenous variables or
a named list, where element \code{coeffs} contains a \eqn{pK^2 \times S} matrix of MCMC coefficient draws
of lagged endogenous variables and element \code{lambda} contains the corresponding draws of inclusion
parameters in case variable selection algorithms were employed. For time varying parameter models
the coefficient matrix must be \eqn{pTK^2 \times S}. Draws of the error covariance matrix of the state
equation can be provided as a \eqn{pK^2 \times S} matrix in an additional list element.}

\item{B}{either a \eqn{((1 + s)MK) \times S} matrix of MCMC coefficient draws of unmodelled, non-deterministic variables
or a named list, where element \code{coeffs} contains a \eqn{((1 + s)MK) \times S} matrix of MCMC coefficient draws of
unmodelled, non-deterministic variables and element \code{lambda} contains the corresponding draws of inclusion
parameters in case variable selection algorithms were employed. For time varying parameter models
the coefficient matrix must be \eqn{(1 + s)TMK \times S}. Draws of the error covariance matrix of the state
equation can be provided as a \eqn{(1 + s)MK \times S} matrix in an additional list element.}

\item{C}{either a \eqn{KN \times S} matrix of MCMC coefficient draws of deterministic terms or
a named list, where element \code{coeffs} contains a \eqn{KN \times S} matrix of MCMC coefficient draws of
deterministic terms and element \code{lambda} contains the corresponding draws of inclusion
parameters in case variable selection algorithms were employed. For time varying parameter models
the coefficient matrix must be \eqn{TKN \times S}. Draws of the error covariance matrix of the state
equation can be provided as a \eqn{KN \times S} matrix in an additional list element.}

\item{Sigma}{a \eqn{K^2 \times S} matrix of MCMC draws for the error variance-covariance matrix or
a named list, where element \code{coeffs} contains a \eqn{K^2 \times S} matrix of MCMC draws for the
error variance-covariance matrix and element \code{lambda} contains the corresponding draws of inclusion
parameters in case variable selection algorithms were employed to the covariances. For time varying parameter models
the coefficient matrix must be \eqn{TK^2 \times S}. Draws of the error covariance matrix of the state
equation can be provided as a \eqn{K^2 \times S} matrix in an additional list element.}

\item{ci}{a numeric between 0 and 1 specifying the probability mass covered by the
credible intervals. Defaults to 0.95.}

\item{type}{either \code{"hist"} (default) for histograms, \code{"trace"} for a trace plot
or \code{"boxplot"} for a boxplot. Only used for parameter draws of constant coefficients.}

\item{...}{additional arguments.}

\item{object}{an object of class \code{"bvar"}, usually, a result of a call to
\code{\link{bvar}} or \code{\link{bvec_to_bvar}}.}

\item{n.ahead}{number of steps ahead at which to predict.}

\item{new_x}{an object of class \code{ts} of new non-deterministic, exogenous variables.
The object must have the same frequency as the time series in \code{object[["x"]]} and must contain
at least all necessary observations for the predicted period.}

\item{new_d}{a matrix of new deterministic variables. Must have \code{n.ahead} rows.}
}
\value{
An object of class \code{"bvar"} containing the following components, if specified:
\item{data}{the original time-series object of endogenous variables.}
\item{exogen}{the original time-series object of unmodelled variables.}
\item{y}{a \eqn{K \times T} matrix of endogenous variables.}
\item{x}{a \eqn{(pK + (1+s)M + N) \times T} matrix of regressor variables.}
\item{A0}{an \eqn{S \times K^2} "mcmc" object of coefficient draws of structural parameters. In case of time varying parameters a list of such objects.}
\item{A0_lambda}{an \eqn{S \times K^2} "mcmc" object of inclusion parameters for structural parameters.}
\item{A0_sigma}{an \eqn{S \times K^2} "mcmc" object of the error covariance matrices of the structural parameters in a model with time varying parameters.}
\item{A}{an \eqn{S \times pK^2} "mcmc" object of coefficient draws of lagged endogenous variables. In case of time varying parameters a list of such objects.}
\item{A_lambda}{an \eqn{S \times pK^2} "mcmc" object of inclusion parameters for lagged endogenous variables.}
\item{A_sigma}{an \eqn{S \times pK^2} "mcmc" object of the error covariance matrices of coefficients of lagged endogenous variables in a model with time varying parameters.}
\item{B}{an \eqn{S \times ((1 + s)MK)} "mcmc" object of coefficient draws of unmodelled, non-deterministic variables. In case of time varying parameters a list of such objects.}
\item{B_lambda}{an \eqn{S \times ((1 + s)MK)} "mcmc" object of inclusion parameters for unmodelled, non-deterministic variables.}
\item{B_sigma}{an \eqn{S \times ((1 + s)MK)} "mcmc" object of the error covariance matrices of coefficients of unmodelled, non-deterministic variables in a model with time varying parameters.}
\item{C}{an \eqn{S \times NK} "mcmc" object of coefficient draws of deterministic terms. In case of time varying parameters a list of such objects.}
\item{C_lambda}{an \eqn{S \times NK} "mcmc" object of inclusion parameters for deterministic terms.}
\item{C_sigma}{an \eqn{S \times NK} "mcmc" object of the error covariance matrices of coefficients of deterministic terms in a model with time varying parameters.}
\item{Sigma}{an \eqn{S \times K^2} "mcmc" object of variance-covariance draws. In case of time varying parameters a list of such objects.}
\item{Sigma_lambda}{an \eqn{S \times K^2} "mcmc" object of inclusion parameters for error covariances.}
\item{Sigma_sigma}{an \eqn{S \times K^2} "mcmc" object of the error covariance matrices of the coefficients of the error covariance matrix of the measurement equation of a model with time varying parameters.}
\item{specifications}{a list containing information on the model specification.}

A time-series object of class \code{"bvarprd"}.
}
\description{
\code{bvar} is used to create objects of class \code{"bvar"}.

A plot function for objects of class \code{"bvar"}.

Forecasting a Bayesian VAR object of class \code{"bvar"} with credible bands.
}
\details{
For the VARX model
\deqn{A_0 y_t = \sum_{i = 1}^{p} A_i y_{t-i} + \sum_{i = 0}^{s} B_i x_{t - i} + C d_t + u_t}
the function collects the S draws of a Gibbs sampler (after the burn-in phase) in a standardised object,
where \eqn{y_t} is a K-dimensional vector of endogenous variables,
\eqn{A_0} is a \eqn{K \times K} matrix of structural coefficients.
\eqn{A_i} is a \eqn{K \times K} coefficient matrix of lagged endogenous variabels.
\eqn{x_t} is an M-dimensional vector of unmodelled, non-deterministic variables
and \eqn{B_i} its corresponding coefficient matrix.
\eqn{d_t} is an N-dimensional vector of deterministic terms
and \eqn{C} its corresponding coefficient matrix.
\eqn{u_t} is an error term with \eqn{u_t \sim N(0, \Sigma_u)}.

For time varying parameter and stochastic volatility models the respective coefficients and
error covariance matrix of the above model are assumed to be time varying, respectively.

The draws of the different coefficient matrices provided in \code{A0}, \code{A},
\code{B}, \code{C} and \code{Sigma} have to correspond to the same MCMC iterations.

For the VAR model
\deqn{A_0 y_t = \sum_{i = 1}^{p} A_{i} y_{t-i} + \sum_{i = 0}^{s} B_{i} x_{t-i} + C D_t + u_t,}
with \eqn{u_t \sim N(0, \Sigma)} the function produces \code{n.ahead} forecasts.
}
\examples{

# Get data
data("e1")
e1 <- diff(log(e1))
e1 <- window(e1, end = c(1978, 4))

# Generate model data
data <- gen_var(e1, p = 2, deterministic = "const")

# Add priors
model <- add_priors(data,
                    coef = list(v_i = 0, v_i_det = 0),
                    sigma = list(df = 0, scale = .00001))

# Set RNG seed for reproducibility 
set.seed(1234567)

iterations <- 400 # Number of iterations of the Gibbs sampler
# Chosen number of iterations and burnin should be much higher.
burnin <- 100 # Number of burn-in draws
draws <- iterations + burnin # Total number of MCMC draws

y <- t(model$data$Y)
x <- t(model$data$Z)
tt <- ncol(y) # Number of observations
k <- nrow(y) # Number of endogenous variables
m <- k * nrow(x) # Number of estimated coefficients

# Priors
a_mu_prior <- model$priors$coefficients$mu # Vector of prior parameter means
a_v_i_prior <- model$priors$coefficients$v_i # Inverse of the prior covariance matrix

u_sigma_df_prior <- model$priors$sigma$df # Prior degrees of freedom
u_sigma_scale_prior <- model$priors$sigma$scale # Prior covariance matrix
u_sigma_df_post <- tt + u_sigma_df_prior # Posterior degrees of freedom

# Initial values
u_sigma_i <- diag(1 / .00001, k)

# Data containers for posterior draws
draws_a <- matrix(NA, m, iterations)
draws_sigma <- matrix(NA, k^2, iterations)

# Start Gibbs sampler
for (draw in 1:draws) {
 # Draw conditional mean parameters
 a <- post_normal(y, x, u_sigma_i, a_mu_prior, a_v_i_prior)

 # Draw variance-covariance matrix
 u <- y - matrix(a, k) \%*\% x # Obtain residuals
 u_sigma_scale_post <- solve(u_sigma_scale_prior + tcrossprod(u))
 u_sigma_i <- matrix(rWishart(1, u_sigma_df_post, u_sigma_scale_post)[,, 1], k)

 # Store draws
 if (draw > burnin) {
  draws_a[, draw - burnin] <- a
  draws_sigma[, draw - burnin] <- solve(u_sigma_i)
 }
}

# Generate bvar object
bvar_est <- bvar(y = model$data$Y, x = model$data$Z,
                 A = draws_a[1:18,], C = draws_a[19:21, ],
                 Sigma = draws_sigma)
                 

# Load data 
data("e1")
e1 <- diff(log(e1)) * 100

# Generate model
model <- gen_var(e1, p = 1, deterministic = 2,
                 iterations = 100, burnin = 10)
# Chosen number of iterations and burn-in should be much higher.

# Add priors
model <- add_priors(model)

# Obtain posterior draws
object <- draw_posterior(model)

# Plot draws
plot(object)


# Load data
data("e1")
e1 <- diff(log(e1)) * 100
e1 <- window(e1, end = c(1978, 4))

# Generate model data
model <- gen_var(e1, p = 0, deterministic = "const",
                 iterations = 100, burnin = 10)
# Chosen number of iterations and burnin should be much higher.

# Add prior specifications
model <- add_priors(model)

# Obtain posterior draws
object <- draw_posterior(model)

# Generate forecasts
bvar_pred <- predict(object, n.ahead = 10, new_d = rep(1, 10))

# Plot forecasts
plot(bvar_pred)

}
\references{
Lütkepohl, H. (2006). \emph{New introduction to multiple time series analysis} (2nd ed.). Berlin: Springer.
}
