\encoding{UTF-8}
\name{map_taxa}
\alias{map_taxa}
\title{Map taxonomic names to NCBI or GTDB taxonomy}
\description{
Maps taxonomic names to NCBI (RefSeq) or GTDB taxonomy by automatic matching of taxonomic names, with manual mappings for some groups.
}

\usage{
  map_taxa(taxacounts = NULL, refdb = "GTDB_220",
    taxon_AA = NULL, quiet = FALSE)
}

\arguments{
  \item{taxacounts}{data frame with taxonomic name and abundances}
  \item{refdb}{character, name of reference database (\samp{GTDB_220} or \samp{RefSeq_206})}
  \item{taxon_AA}{data frame, amino acid compositions of taxa, used to bypass \code{refdb} specification}
  \item{quiet}{logical, suppress printed messages?}
}

\details{
This function maps taxonomic names to the NCBI (RefSeq) or GTDB taxonomy.
\code{taxacounts} should be a data frame generated by either \code{\link{read_RDP}} or \code{\link{ps_taxacounts}}.
Input names are made by combining the taxonomic rank and name with an underscore separator (e.g. \samp{genus_ Escherichia/Shigella}).
Input names are then matched to the taxa listed in \file{taxon_AA.csv.xz} found under \file{RefDB/RefSeq_206} or \file{RefDB/GTDB_220}.
The \code{protein} and \code{organism} columns in these files hold the rank and taxon name extracted from the RefSeq or GTDB database.
Only exactly matching names are automatically mapped.

For mapping to the NCBI (RefSeq) taxonomy, some group names are manually mapped as follows (see Dick and Tan, 2023):
\tabular{ll}{
    \strong{RDP training set} \tab \strong{NCBI} \cr
    genus_Escherichia/Shigella \tab genus_Escherichia \cr
    phylum_Cyanobacteria/Chloroplast \tab phylum_Cyanobacteria \cr
    genus_Marinimicrobia_genera_incertae_sedis \tab species_Candidatus Marinimicrobia bacterium \cr
    class_Cyanobacteria \tab phylum_Cyanobacteria \cr
    genus_Spartobacteria_genera_incertae_sedis \tab species_Spartobacteria bacterium LR76 \cr
    class_Planctomycetacia \tab class_Planctomycetia \cr
    class_Actinobacteria \tab phylum_Actinobacteria \cr
    order_Rhizobiales \tab order_Hyphomicrobiales \cr
    genus_Gp1 \tab genus_Acidobacterium \cr
    genus_Gp6 \tab genus_Luteitalea \cr
    genus_GpI \tab genus_Nostoc \cr
    genus_GpIIa \tab genus_Synechococcus \cr
    genus_GpVI \tab genus_Pseudanabaena \cr
    family_Family II \tab family_Synechococcaceae \cr
    genus_Subdivision3_genera_incertae_sedis \tab family_Verrucomicrobia subdivision 3 \cr
    order_Clostridiales \tab order_Eubacteriales \cr
    family_Ruminococcaceae \tab family_Oscillospiraceae
}

To avoid manual mapping, GTDB can be used for both taxonomic assignemnts and reference proteomes.
16S rRNA sequences from GTDB release 220 are available for the RDP Classifier (\doi{10.5281/zenodo.7633099}) and \pkg{dada2} (\doi{10.5281/zenodo.13984843}).
Example files created using the RDP Classifier are provided under \file{extdata/RDP-GTDB_220}.
An example dataset created with DADA2 is \code{data(\link{mouse.GTDB_220})}; this is a \code{\link[phyloseq]{phyloseq-class}} object that can be processed with functions described at \code{\link{physeq}}.

Change \code{quiet} to TRUE to suppress printing of messages about manual mappings, most abundant unmapped groups, and overall percentage of mapped names.
}

\value{
Integer vector with length equal to number of rows of \code{taxacounts}.
Values are rownumbers in the data frame generated by reading \code{taxon_AA.csv.xz}, or NA for no matching taxon.
Attributes \code{unmapped_groups} and \code{unmapped_percent} have the input names of unmapped groups and their percentage of the total classification count.
}

\examples{
# Partial mapping from RDP training set to NCBI taxonomy
file <- system.file("extdata/RDP/SMS+12.tab.xz", package = "chem16S")
# Use drop.groups = TRUE to exclude root- and domain-level
# classifications and certain non-prokaryotic groups
RDP <- read_RDP(file, drop.groups = TRUE)
map <- map_taxa(RDP, refdb = "RefSeq_206")
# About 24% of classifications are unmapped
sum(attributes(map)$unmapped_percent)

# 100% mapping from GTDB training set to GTDB taxonomy
file <- system.file("extdata/RDP-GTDB_220/SMS+12.tab.xz", package = "chem16S")
RDP.GTDB <- read_RDP(file)
map.GTDB <- map_taxa(RDP.GTDB)
stopifnot(all.equal(sum(attributes(map.GTDB)$unmapped_percent), 0))
}

\references{
Dick JM, Tan J. 2023. Chemical links between redox conditions and estimated community proteomes from 16S rRNA and reference protein sequences. \emph{Microbial Ecology} \bold{85}: 1338--1355. \doi{10.1007/s00248-022-01988-9}
}
