#' Plot the density estimate obtained from grouped summary statistics using \link{degross} and superpose it to the observed histogram.
#'
#' @description Plot the density estimate corresponding to a \code{degross} object and superpose it to the observed histogram.
#'
#' @usage
#' \method{plot}{degross}(x, col="black", lwd=2, lty="dashed", xlab="", ylab="Density", main="",...)
#'
#' @param x A \link{degross.object} generated by \link{degross}.
#' @param col Color used for plotting the fitted density.
#' @param lwd Line width for the fitted density curve.
#' @param lty Line type for the the fitted density curve.
#' @param xlab Label on the x-axis.
#' @param ylab Label on the y-axis.
#' @param main Title for the generated graph.
#' @param ...  Further arguments to be passed to \code{hist}.
#'
#' @return A histogram based on the observed big bin frequencies with the fitted density superposed.
#'
#' @author Philippe Lambert \email{p.lambert@uliege.be}
#' @references
#' Lambert, P. (2021) Moment-based density and risk estimation from grouped summary statistics. arXiv:2107.03883.
#'
#' @seealso \code{\link{degross}}, \code{\link{degross.object}}, \code{\link{print.degross}}.
#'
#' @examples
#' sim = simDegrossData(n=3500, plotting=TRUE,choice=2) ## Generate grouped data
#' obj.data = degrossData(Big.bins=sim$Big.bins, freq.j=sim$freq.j, m.j=sim$m.j)
#' print(obj.data)
#' obj.fit = degross(obj.data) ## Estimate the underlying density
#' plot(obj.fit) ## Plot the fitted density with the data histogram
#'
#' @export
plot.degross = function(x,col="black",lwd=2,lty="dashed",xlab="",ylab="Density",main="",...){
  freq.j = x$degross.data$freq.j
  Big.bins = x$degross.data$Big.bins
  Big.mids = .5*(Big.bins[-1] + Big.bins[-length(Big.bins)])
  temp = hist(rep(Big.mids,freq.j),breaks=Big.bins,plot=FALSE)
  hist(rep(Big.mids,freq.j), ylim=c(0,1.1*max(temp$density, x$pi.i/x$delta)),
       breaks=Big.bins,freq=FALSE,
       col="grey85",border="white",
       xlab=xlab,ylab=ylab,main=main,...)
  with(x, lines(ui,pi.i/delta,col=col,lwd=lwd,lty=lty))
}
